<?php

namespace App\Http\Controllers;

use App\Models\Deal;
use App\Models\DealProduct;
use App\Services\ShopifyService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\View\View;

class DealsController extends Controller
{
    public function __construct(
        protected ShopifyService $shopify
    ) {}

    public function index(): View
    {
        $deals = Deal::with('dealProducts')->latest()->paginate(10);
        return view('deals.index', compact('deals'));
    }

    /**
     * Search Shopify products by title (API).
     */
    public function search(Request $request): JsonResponse
    {
        $request->validate(['q' => 'nullable|string|max:200']);
        $q = $request->input('q', '');
        $query = $q !== '' ? 'title:*' . preg_replace('/\s+/', '* ', trim($q)) . '*' : 'status:active';

        try {
            $result = $this->shopify->searchProductsByTitle($query, 50);
        } catch (\Throwable $e) {
            return response()->json(['error' => $e->getMessage()], 502);
        }

        $products = [];
        foreach ($result['edges'] ?? [] as $edge) {
            $node = $edge['node'];
            $variants = $node['variants']['edges'] ?? [];
            foreach ($variants as $ve) {
                $v = $ve['node'];
                $products[] = [
                    'product_id' => $node['id'],
                    'variant_id' => $v['id'],
                    'title' => $node['title'] ?? '',
                    'price' => $v['price'] ?? '0',
                    'compare_at_price' => $v['compareAtPrice'] ?? null,
                ];
            }
        }

        return response()->json([
            'products' => $products,
            'pageInfo' => $result['pageInfo'] ?? [],
        ]);
    }

    /**
     * Apply deal: save original prices in DB, update Shopify (rule: -$1), create deal with duration.
     */
    public function apply(Request $request): JsonResponse
    {
        $request->validate([
            'items' => 'required|array',
            'items.*.variant_id' => 'required|string',
            'items.*.product_id' => 'nullable|string',
            'items.*.title' => 'nullable|string',
            'items.*.price' => 'required|string',
            'items.*.compare_at_price' => 'nullable|string',
            'duration_value' => 'required|integer|min:1',
            'duration_unit' => 'required|in:hours,days',
        ]);

        $durationValue = (int) $request->input('duration_value');
        $durationUnit = $request->input('duration_unit');
        $hours = $durationUnit === 'days' ? $durationValue * 24 : $durationValue;

        $deal = Deal::create([
            'duration_value' => $durationValue,
            'duration_unit' => $durationUnit,
            'starts_at' => now(),
            'ends_at' => now()->addHours($hours),
            'restored' => false,
        ]);

        $items = $request->input('items');
        $errors = [];

        foreach ($items as $item) {
            $variantId = $item['variant_id'];
            $price = $item['price'];
            $compareAt = $item['compare_at_price'] ?? null;

            $originalPrice = (float) $price;
            $originalCompare = $compareAt !== null && $compareAt !== '' ? (float) $compareAt : null;

            $dealPrice = max(0, $originalPrice - 1);
            $dealCompare = $originalCompare !== null ? max(0, $originalCompare - 1) : null;

            DealProduct::create([
                'deal_id' => $deal->id,
                'shopify_product_id' => $item['product_id'] ?? null,
                'shopify_variant_id' => $variantId,
                'title' => $item['title'] ?? null,
                'original_price' => $originalPrice,
                'original_compare_at_price' => $originalCompare,
                'deal_price' => $dealPrice,
                'deal_compare_at_price' => $dealCompare,
                'restored' => false,
            ]);

            try {
                $this->shopify->updateVariantPrice(
                    $variantId,
                    (string) $dealPrice,
                    $dealCompare !== null ? (string) $dealCompare : null
                );
            } catch (\Throwable $e) {
                $errors[] = $item['title'] . ': ' . $e->getMessage();
            }
        }

        return response()->json([
            'success' => true,
            'deal_id' => $deal->id,
            'message' => count($errors) === 0
                ? 'Deal applied. Prices will be restored at ' . $deal->ends_at->toDateTimeString()
                : 'Deal saved with some Shopify errors: ' . implode('; ', $errors),
        ]);
    }
}
