<?php

namespace App\Service;

use GuzzleHttp\Client;
use Illuminate\Support\Facades\Log;

class ApiService
{
    protected $client;
    protected $baseUrl;
    protected $token;

    /**
     * Constructor para inicializar la URL base y el token de autenticación.
     */
    public function __construct()
    {
        $this->baseUrl = config('services.api.base_url'); // Configurar en config/services.php
        $this->token = config('services.api.token'); // Configurar en config/services.php
        Log::info("baseUrl", ["response" => $this->baseUrl]);
        $this->client = new Client([
            'base_uri' => $this->baseUrl,
            'headers' => [
                'Authorization' => 'Bearer ' . $this->token,
                'Accept' => 'application/json',
            ],
        ]);
    }

    /**
     * Método para realizar una solicitud GET.
     *
     * @param string $endpoint
     * @param array $queryParams
     * @return array|null
     */
    public function get(string $endpoint, array $queryParams = []): ?array
    {
        //dd($this->client);
        $response = $this->client->get($endpoint, ['query' => $queryParams]);

        Log::info("response", ["response" => $response]);
        return $this->handleResponse($response);
    }

    /**
     * Método para realizar una solicitud POST.
     *
     * @param string $endpoint
     * @param array $data
     * @return array|null
     */
    public function post(string $endpoint, array $data = []): ?array
    {
        $response = $this->client->post($endpoint, ['json' => $data]);
        return $this->handleResponse($response);
    }

    /**
     * Método para manejar la respuesta de la API.
     *
     * @param \Psr\Http\Message\ResponseInterface $response
     * @return array|null
     */
    private function handleResponse($response): ?array
    {
        $statusCode = $response->getStatusCode();
        $body = $response->getBody()->getContents();

        if ($statusCode >= 200 && $statusCode < 300) {
            return json_decode($body, true);
        }

        // Manejo de errores (puedes personalizar esto).
        throw new \Exception('Error al consumir la API: ' . $body);
    }
}
