<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;

class GoldPriceWeightRule extends Model
{
    use HasFactory;

    protected $fillable = [
        'collection_id',
        'min_weight',
        'max_weight',
        'fee_10k',
        'fee_14k',
        'order',
    ];

    protected $casts = [
        'min_weight' => 'decimal:3',
        'max_weight' => 'decimal:3',
        'fee_10k' => 'decimal:2',
        'fee_14k' => 'decimal:2',
        'order' => 'integer',
    ];

    /**
     * Scope to get rules ordered by min_weight
     */
    public function scopeOrdered(Builder $query): Builder
    {
        return $query->orderBy('min_weight', 'asc');
    }

    /**
     * Scope to get rules for a specific collection
     */
    public function scopeForCollection(Builder $query, string $collectionId): Builder
    {
        return $query->where('collection_id', $collectionId);
    }

    /**
     * Check if a weight falls within this rule's range
     */
    public function matchesWeight(float $weight): bool
    {
        return $weight >= (float)$this->min_weight && $weight <= (float)$this->max_weight;
    }

    /**
     * Check if this rule overlaps with another rule
     */
    public function overlapsWith(GoldPriceWeightRule $other): bool
    {
        // Check if ranges overlap
        // Two ranges overlap if: min1 <= max2 && min2 <= max1
        return (float)$this->min_weight <= (float)$other->max_weight 
            && (float)$other->min_weight <= (float)$this->max_weight;
    }

    /**
     * Find the rule that matches a given weight for a specific collection
     */
    public static function findRuleForWeight(string $collectionId, float $weight): ?self
    {
        return static::forCollection($collectionId)
            ->ordered()
            ->where('min_weight', '<=', $weight)
            ->where('max_weight', '>=', $weight)
            ->first();
    }
}
