<?php

namespace App\Jobs;

use App\Actions\ListShopifyCollectionsAction;
use App\Actions\UpdateShopifyCollectionSalePriceFrontGold;
use App\Actions\UpdateShopifyCollectionReferencePrice;
use App\Actions\UpdateShopifyCollectionReferencePriceFromSale;
use App\Actions\GetMaterialDataByCollectionAction;
use App\Actions\FetchGoldPriceAction;
use App\Helpers\GoldPriceHelper;
use App\Jobs\SendGoldPriceUpdateEmail;
use App\Models\Configuration;
use App\Models\GoldPrice;
use App\Models\GoldPriceLog;
use App\Models\GoldPriceErrorLog;
use App\Models\GoldPriceLogDetail;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

class UpdateGoldPriceJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $executionType;
    protected $collectionName;

    /**
     * Create a new job instance.
     */
    public function __construct($executionType = 'automatic', $collectionName = null)
    {
        $this->executionType = $executionType;
        $this->collectionName = $collectionName; // Can be null to process all enabled collections
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        // Prevent "Maximum execution time of 30 seconds exceeded" when processing many collections/variants
        set_time_limit(0);

        Log::info('UpdateGoldPriceJob: Job started', [
            'execution_type' => $this->executionType,
            'collection_name' => $this->collectionName,
        ]);

        // Use a lock to prevent multiple instances from running simultaneously
        // TTL 10 min: if job crashes without releasing, lock expires so next run can proceed
        $lockSeconds = 600;
        $lock = Cache::lock('update_gold_price_job', $lockSeconds);
        $lockAcquired = false;

        try {
            // Try to acquire the lock
            if (!$lock->get()) {
                Log::warning('UpdateGoldPriceJob: Lock not acquired, skipping execution', [
                    'hint' => 'Another instance may be running, or a previous run crashed without releasing the lock. Lock expires in ' . round($lockSeconds / 60) . ' minutes.',
                ]);
                return;
            }

            $lockAcquired = true;
            Log::info('UpdateGoldPriceJob: Lock acquired, starting executeUpdate');

            // Lock acquired, proceed with execution
            $this->executeUpdate();
        } catch (\Exception $e) {
            Log::error('UpdateGoldPriceJob: Error acquiring lock or during execution', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            throw $e;
        } finally {
            // Always release the lock if it was acquired
            if ($lockAcquired) {
                try {
                    $lock->release();
                } catch (\Exception $releaseException) {
                    Log::warning('UpdateGoldPriceJob: Error releasing lock', [
                        'error' => $releaseException->getMessage()
                    ]);
                }
            }
        }
    }

    /**
     * Execute the actual update process.
     */
    protected function executeUpdate(): void
    {
        $startTime = now();
        Log::info('UpdateGoldPriceJob: executeUpdate started');

        // Get configuration
        $config = Configuration::getInstance();
        Log::info('UpdateGoldPriceJob: Config loaded', [
            'auto_10k' => $config->auto_10k,
            'auto_14k' => $config->auto_14k,
        ]);

        // Fetch latest gold price from API before updating
        try {
            $fetchAction = new FetchGoldPriceAction();
            $latestGoldPrice = $fetchAction();
            Log::info('UpdateGoldPriceJob: Gold price fetched', [
                'price' => $latestGoldPrice->price ?? null,
            ]);
        } catch (\Throwable $fetchError) {
            Log::warning('Failed to fetch gold price from API, trying to use latest from database', [
                'error' => $fetchError->getMessage()
            ]);
            // Fallback: try to use latest price from database
            $latestGoldPrice = GoldPrice::latest('created_at')->first();
            if (!$latestGoldPrice) {
                Log::error('No gold price found in database and API fetch failed');
                return;
            }
        }

        // Create log entry (will be updated with final collection name later)
        $log = GoldPriceLog::create([
            'execution_type' => $this->executionType,
            'collection_name' => $this->collectionName,
            'gold_price_per_gram' => $latestGoldPrice->price / 31.1035, // Default calculation for log
            'status' => 'success',
            'started_at' => $startTime,
        ]);


        // Check if auto-update is enabled
        if (!$config->auto_10k && !$config->auto_14k) {
            Log::info('UpdateGoldPriceJob: Exiting - auto_10k and auto_14k disabled');
            $log->update([
                'status' => 'success',
                'summary_message' => 'Auto-update is disabled for both 10k and 14k',
                'completed_at' => now(),
                'duration_seconds' => now()->diffInSeconds($startTime),
            ]);
            return;
        }

        // Get collections to process
        $collectionsToProcess = [];

        if ($this->collectionName) {
            // Process specific collection
            $collectionsToProcess = [['name' => $this->collectionName, 'enabled' => true]];
            Log::info('UpdateGoldPriceJob: Using specific collection', ['name' => $this->collectionName]);
        } else {
            // Process all enabled collections from config
            $configuredCollections = $config->collections ?? [];
            foreach ($configuredCollections as $collectionConfig) {
                if ($collectionConfig['enabled'] ?? false) {
                    $collectionsToProcess[] = $collectionConfig;
                }
            }
            Log::info('UpdateGoldPriceJob: Using enabled collections from config', [
                'count' => count($collectionsToProcess),
                'names' => array_column($collectionsToProcess, 'name'),
            ]);
        }

        if (empty($collectionsToProcess)) {
            Log::warning('UpdateGoldPriceJob: No collections to process, exiting');
            $log->update([
                'status' => 'success',
                'summary_message' => 'No enabled collections found to process',
                'completed_at' => now(),
                'duration_seconds' => now()->diffInSeconds($startTime),
            ]);
            return;
        }

        // Process each collection
        $totalUpdated10k = 0;
        $totalUpdated14k = 0;
        $totalErrors10k = 0;
        $totalErrors14k = 0;
        $totalProductsProcessed = 0;

        foreach ($collectionsToProcess as $collectionToProcess) {
            $collectionName = $collectionToProcess['name'];

            // Get collection by name from Shopify
            $collectionsData = (new ListShopifyCollectionsAction())($collectionName);
            $collections = $collectionsData['collections'] ?? [];

            if (empty($collections)) {
                Log::error('UpdateGoldPriceJob: Collection not found in Shopify', ['collection' => $collectionName]);
                continue;
            }

            $collection = $collections[0];
            $collectionId = is_array($collection) ? $collection['id'] : $collection->id;
            $collectionTitle = is_array($collection) ? $collection['title'] : $collection->title;
            Log::info('UpdateGoldPriceJob: Collection resolved', [
                'name' => $collectionName,
                'id' => $collectionId,
                'title' => $collectionTitle,
            ]);

            $updated10k = 0;
            $updated14k = 0;
            $errors10k = 0;
            $errors14k = 0;
            $price10kUsed = null;
            $price14kUsed = null;

            try {
                // Get material data for this collection
                $materialData = (new GetMaterialDataByCollectionAction())($collectionId);

                $totalProductsProcessed += $materialData->total;
                Log::info('UpdateGoldPriceJob: Material data for collection', [
                    'collection' => $collectionName,
                    'total_products' => $materialData->total,
                    'count_10K' => $materialData->count_10K,
                    'count_14K' => $materialData->count_14K,
                ]);

                // Update 10k if enabled and collection has 10k products
                if ($config->auto_10k && $materialData->count_10K) {
                    // Calculate gold price per gram for 10k in this collection using the helper (without weight for display)
                    // The actual price per gram will be calculated per variant using weight rules
                    $goldPricePerGram10k = GoldPriceHelper::calculateGoldPricePerGram($collectionName, '10k', $latestGoldPrice->price);
                    $goldPriceRounded10k = round($goldPricePerGram10k);
                    // Store rounded price (same as used for updating) for email consistency
                    $price10kUsed = $goldPriceRounded10k;

                    try {
                        // Use the current collection ID (already fetched above)
                        $collectionIdForUpdate = $collectionId;

                        $request = new \Illuminate\Http\Request();
                        $request->merge([
                            'collection_id' => $collectionIdForUpdate,
                            'collection_name' => $collectionName,
                            'material' => '10k',
                            'rounding_option' => $config->rounding_option ?? 'default', // Use rounding option from configuration
                            // Don't pass 'price' to allow weight-based rules calculation
                        ]);

                        // Call the update action and capture errors
                        // Pass false as second parameter to prevent sending individual emails (job will send one at the end)
                        try {
                            Log::info('UpdateGoldPriceJob: Calling UpdateShopifyCollectionSalePriceFrontGold for 10k', ['collection' => $collectionName]);
                            $result = (new UpdateShopifyCollectionSalePriceFrontGold())($request, false);
                            // Extract real count of updated products from the action result
                            // UpdateShopifyCollectionSalePriceFrontGold always returns a Laravel Response object
                            if ($result && $result->getStatusCode() === 200) {
                                $resultData = json_decode($result->getContent(), true);
                                if (isset($resultData['data']['products_updated'])) {
                                    $updated10k = (int)$resultData['data']['products_updated']; // Products updated (not variants)
                                }
                                if (isset($resultData['data']['variants_errors'])) {
                                    $errors10k += (int)$resultData['data']['variants_errors'];
                                }
                                Log::info('UpdateGoldPriceJob: 10k update result', [
                                    'collection' => $collectionName,
                                    'products_updated' => $updated10k,
                                    'variants_errors' => $errors10k,
                                    'raw_response_data' => $resultData['data'] ?? null,
                                ]);
                            } else {
                                Log::warning('UpdateGoldPriceJob: 10k update returned non-200 or null', [
                                    'collection' => $collectionName,
                                    'status_code' => $result ? $result->getStatusCode() : null,
                                ]);
                            }
                        } catch (\Exception $e) {
                            $errors10k++;

                            GoldPriceErrorLog::create([
                                'gold_price_log_id' => $log->id,
                                'material_type' => '10k',
                                'error_message' => $e->getMessage(),
                                'error_trace' => $e->getTraceAsString(),
                            ]);

                            Log::error('UpdateGoldPriceJob: Error updating 10k', ['collection' => $collectionName, 'error' => $e->getMessage()]);
                        }
                    } catch (\Exception $e) {
                        $errors10k++;
                        Log::error('UpdateGoldPriceJob: Error processing 10k', ['collection' => $collectionName, 'error' => $e->getMessage()]);
                    }

                    // updated10k is now set from the actual result above
                } else {
                    Log::info('UpdateGoldPriceJob: Skipping 10k update (disabled or no 10k products)', [
                        'collection' => $collectionName,
                        'auto_10k' => $config->auto_10k,
                        'count_10K' => $materialData->count_10K ?? 0,
                    ]);
                }

                // Update 14k if enabled and collection has 14k products
                if ($config->auto_14k && $materialData->count_14K) {
                    // Calculate gold price per gram for 14k in this collection using the helper (without weight for display)
                    // The actual price per gram will be calculated per variant using weight rules
                    $goldPricePerGram14k = GoldPriceHelper::calculateGoldPricePerGram($collectionName, '14k', $latestGoldPrice->price);
                    $goldPriceRounded14k = round($goldPricePerGram14k);
                    // Store rounded price (same as used for updating) for email consistency
                    $price14kUsed = $goldPriceRounded14k;

                    try {
                        // Use the current collection ID (already fetched above)
                        $collectionIdForUpdate = $collectionId;

                        $request = new \Illuminate\Http\Request();
                        $request->merge([
                            'collection_id' => $collectionIdForUpdate,
                            'collection_name' => $collectionName,
                            'material' => '14k',
                            'rounding_option' => $config->rounding_option ?? 'default', // Use rounding option from configuration
                            // Don't pass 'price' to allow weight-based rules calculation
                        ]);

                        // Call the update action and capture errors
                        // Pass false as second parameter to prevent sending individual emails (job will send one at the end)
                        try {
                            Log::info('UpdateGoldPriceJob: Calling UpdateShopifyCollectionSalePriceFrontGold for 14k', ['collection' => $collectionName]);
                            $result = (new UpdateShopifyCollectionSalePriceFrontGold())($request, false);
                            // Extract real count of updated products from the action result
                            // UpdateShopifyCollectionSalePriceFrontGold always returns a Laravel Response object
                            if ($result && $result->getStatusCode() === 200) {
                                $resultData = json_decode($result->getContent(), true);
                                if (isset($resultData['data']['products_updated'])) {
                                    $updated14k = (int)$resultData['data']['products_updated']; // Products updated (not variants)
                                }
                                if (isset($resultData['data']['variants_errors'])) {
                                    $errors14k += (int)$resultData['data']['variants_errors'];
                                }
                                Log::info('UpdateGoldPriceJob: 14k update result', [
                                    'collection' => $collectionName,
                                    'products_updated' => $updated14k,
                                    'variants_errors' => $errors14k,
                                    'raw_response_data' => $resultData['data'] ?? null,
                                ]);
                            } else {
                                Log::warning('UpdateGoldPriceJob: 14k update returned non-200 or null', [
                                    'collection' => $collectionName,
                                    'status_code' => $result ? $result->getStatusCode() : null,
                                ]);
                            }
                        } catch (\Exception $e) {
                            $errors14k++;

                            GoldPriceErrorLog::create([
                                'gold_price_log_id' => $log->id,
                                'material_type' => '14k',
                                'error_message' => $e->getMessage(),
                                'error_trace' => $e->getTraceAsString(),
                            ]);

                            Log::error('UpdateGoldPriceJob: Error updating 14k', ['collection' => $collectionName, 'error' => $e->getMessage()]);
                        }
                    } catch (\Exception $e) {
                        $errors14k++;
                        Log::error('UpdateGoldPriceJob: Error processing 14k', ['collection' => $collectionName, 'error' => $e->getMessage()]);
                    }

                    // updated14k is now set from the actual result above
                } else {
                    Log::info('UpdateGoldPriceJob: Skipping 14k update (disabled or no 14k products)', [
                        'collection' => $collectionName,
                        'auto_14k' => $config->auto_14k,
                        'count_14K' => $materialData->count_14K ?? 0,
                    ]);
                }

                // Update compare prices for this collection using saved configuration (no only_draft)
                $compareConfig = $config->compare_price_config ?? null;
                if ($compareConfig && !empty($compareConfig['calculation_method']) && !empty($compareConfig['rounding_option'])) {
                    $roundingOption = $compareConfig['rounding_option'];
                    try {
                        if (($compareConfig['calculation_method'] ?? '') === 'compare_price') {
                            $action = $compareConfig['action'] ?? null;
                            $operation = $compareConfig['operation'] ?? null;
                            $price = isset($compareConfig['price']) ? (float) $compareConfig['price'] : null;
                            if ($action && $operation && $price !== null && $price >= 0) {
                                $req = new \Illuminate\Http\Request();
                                $req->merge([
                                    'collection_id' => $collectionId,
                                    'action' => $action,
                                    'operation' => $operation,
                                    'price' => $price,
                                    'rounding_option' => $roundingOption,
                                    'demo' => false,
                                    'only_draft' => false,
                                ]);
                                (new UpdateShopifyCollectionReferencePrice())($req);
                            }
                        } elseif (($compareConfig['calculation_method'] ?? '') === 'sale_price') {
                            $calculationType = $compareConfig['calculation_type'] ?? null;
                            $amount = isset($compareConfig['amount']) ? (float) $compareConfig['amount'] : null;
                            if ($calculationType && $amount !== null && $amount >= 0 && ($calculationType !== 'discount_based' || $amount < 100)) {
                                $req = new \Illuminate\Http\Request();
                                $req->merge([
                                    'collection_id' => $collectionId,
                                    'amount' => $amount,
                                    'calculation_type' => $calculationType,
                                    'rounding_option' => $roundingOption,
                                    'demo' => false,
                                    'only_draft' => false,
                                ]);
                                (new UpdateShopifyCollectionReferencePriceFromSale())($req);
                            }
                        }
                    } catch (\Exception $e) {
                        Log::warning('Compare price update failed for collection', [
                            'collection' => $collectionName,
                            'error' => $e->getMessage(),
                        ]);
                    }
                }

                // Accumulate totals for all collections
                $totalUpdated10k += $updated10k;
                $totalUpdated14k += $updated14k;
                $totalErrors10k += $errors10k;
                $totalErrors14k += $errors14k;

                // Calculate products_processed as the sum of products updated for 10k and 14k
                // This ensures: products_processed = products_updated_10k + products_updated_14k
                $productsProcessed = $updated10k + $updated14k;

                // Create detail record for this collection
                GoldPriceLogDetail::create([
                    'gold_price_log_id' => $log->id,
                    'collection_name' => $collectionTitle,
                    'collection_id' => $collectionId,
                    'price_10k' => $price10kUsed,
                    'price_14k' => $price14kUsed,
                    'products_processed' => $productsProcessed, // Sum of products updated 10k + 14k
                    'products_updated_10k' => $updated10k,
                    'products_updated_14k' => $updated14k,
                    'errors_10k' => $errors10k,
                    'errors_14k' => $errors14k,
                ]);
                Log::info('UpdateGoldPriceJob: Collection processed', [
                    'collection' => $collectionName,
                    'products_updated_10k' => $updated10k,
                    'products_updated_14k' => $updated14k,
                    'errors_10k' => $errors10k,
                    'errors_14k' => $errors14k,
                ]);
            } catch (\Exception $e) {
                Log::error('UpdateGoldPriceJob: Error processing collection', ['collection' => $collectionName ?? 'unknown', 'error' => $e->getMessage()]);
            }
        }

        // Update log collection name
        $log->update([
            'collection_name' => implode(', ', array_map(function ($c) {
                return $c['name'];
            }, $collectionsToProcess))
        ]);

        // Determine final status
        $updated10k = $totalUpdated10k;
        $updated14k = $totalUpdated14k;
        $errors10k = $totalErrors10k;
        $errors14k = $totalErrors14k;
        $totalErrors = $errors10k + $errors14k;
        $status = 'success';
        if ($totalErrors > 0 && ($updated10k > 0 || $updated14k > 0)) {
            $status = 'partial_success';
        } elseif ($totalErrors > 0) {
            $status = 'failed';
        }

        $endTime = now();
        $duration = $endTime->diffInSeconds($startTime);

        // Update log with final results
        $summaryMessage = sprintf(
            "Successfully updated %d 10k product%s with %d error%s and %d 14k product%s with %d error%s",
            $updated10k,
            $updated10k != 1 ? 's' : '',
            $errors10k,
            $errors10k != 1 ? 's' : '',
            $updated14k,
            $updated14k != 1 ? 's' : '',
            $errors14k,
            $errors14k != 1 ? 's' : ''
        );

        $log->update([
            'total_10k_updated' => $updated10k,
            'total_14k_updated' => $updated14k,
            'total_10k_errors' => $errors10k,
            'total_14k_errors' => $errors14k,
            'status' => $status,
            'summary_message' => $summaryMessage,
            'error_details' => null,
            'completed_at' => $endTime,
            'duration_seconds' => $duration,
            'total_products_processed' => $totalProductsProcessed,
        ]);

        Log::info('UpdateGoldPriceJob: executeUpdate completed', [
            'status' => $status,
            'total_10k_updated' => $updated10k,
            'total_14k_updated' => $updated14k,
            'total_10k_errors' => $errors10k,
            'total_14k_errors' => $errors14k,
            'duration_seconds' => $duration,
            'gold_price_log_id' => $log->id,
        ]);

        // Prepare email data
        $emailCollections = [];
        foreach ($collectionsToProcess as $collectionToProcess) {
            $collectionName = $collectionToProcess['name'];
            $detail = GoldPriceLogDetail::where('gold_price_log_id', $log->id)
                ->where('collection_name', $collectionName)
                ->first();

            if ($detail) {
                $emailCollections[] = [
                    'name' => $detail->collection_name,
                    'products_processed' => $detail->products_processed,
                    'price_10k' => $detail->price_10k,
                    'price_14k' => $detail->price_14k,
                    'products_updated_10k' => $detail->products_updated_10k,
                    'products_updated_14k' => $detail->products_updated_14k,
                    'errors_10k' => $detail->errors_10k,
                    'errors_14k' => $detail->errors_14k,
                    'has_errors' => $detail->errors_10k > 0 || $detail->errors_14k > 0,
                    'error_details' => [],
                ];
            }
        }

        // Dispatch email job asynchronously
        Log::info('UpdateGoldPriceJob: Dispatching email', ['collections_count' => count($emailCollections)]);
        SendGoldPriceUpdateEmail::dispatch(
            $this->executionType,
            $emailCollections,
            $latestGoldPrice->price
        );
    }
}
