<?php

namespace App\Jobs;

use App\Mail\GoldPriceUpdateNotification;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;

class SendGoldPriceUpdateEmail implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public $executionType;
    public $collections;
    public $goldPrice;

    /**
     * Create a new job instance.
     */
    public function __construct($executionType, $collections, $goldPrice)
    {
        $this->executionType = $executionType;
        $this->collections = $collections;
        $this->goldPrice = $goldPrice;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        try {
            $email = 'info@avijewels.com';

            Log::info("Sending gold price update email", [
                'to' => $email,
                'execution_type' => $this->executionType,
                'collections_count' => count($this->collections ?? []),
                'gold_price' => $this->goldPrice
            ]);

            Mail::to($email)->send(
                new GoldPriceUpdateNotification(
                    $this->executionType,
                    $this->collections,
                    $this->goldPrice
                )
            );

            Log::info("Gold price update email sent successfully", [
                'to' => $email,
                'execution_type' => $this->executionType
            ]);
        } catch (\Exception $e) {
            Log::error("Failed to send gold price update email", [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            throw $e; // Re-throw to let Laravel handle job retries
        }
    }
}
