<?php

namespace App\Http\Controllers;

use App\Actions\ListShopifyCollectionsAction;
use App\Actions\UpdateShopifyCollectionSalePrice;
use App\Actions\UpdateShopifyCollectionSalePriceFrontGold;
use App\Actions\UpdateShopifyCollectionCostPrice;
use App\Jobs\UpdateCollectionSalePriceJob;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use Inertia\Inertia;

class ShopifySalePriceController extends Controller
{
    //
    public function listProducts()
    {

        $data = (new ListShopifyCollectionsAction())();


        return Inertia::render('sale-price', ["collections" => $data["collections"], "total" => $data["total"]]);
    }

    public function updatePriceFrontGold(Request $request)
    {
        try {
            // Always process asynchronously with progress tracking (both demo and non-demo)
            $collection_id = $request->input('collection_id');
            $collection_name = $request->input('collection_name');
            $material = $request->input('material');
            $onlyDraft = $request->input('only_draft', false);
            $selectedProductIds = $request->input('selected_product_ids');

            // Mark this request as coming from /price page
            $request->merge(['_from_price_page' => true]);
            
            // Fetch collection data first
            $action = new UpdateShopifyCollectionCostPrice();
            $shopifyData = $action->fetchCollectionData($collection_id);

            // Filter by selected products if provided
            if ($selectedProductIds && is_array($selectedProductIds) && count($selectedProductIds) > 0) {
                $shopifyData['products'] = array_filter($shopifyData['products'], function ($item) use ($selectedProductIds) {
                    return in_array($item['node']['id'], $selectedProductIds);
                });
                $shopifyData['products'] = array_values($shopifyData['products']);
            }

            // Filter by draft status if only_draft is enabled
            if ($onlyDraft) {
                $shopifyData['products'] = array_filter($shopifyData['products'], function ($item) {
                    $status = $item['node']['status'] ?? null;
                    return strtoupper($status) === 'DRAFT';
                });
                $shopifyData['products'] = array_values($shopifyData['products']);
            }

            // Calculate total variants matching the material
            $salePriceAction = new UpdateShopifyCollectionSalePriceFrontGold();
            $totalVariants = $salePriceAction->calculateTotalVariants(
                $shopifyData['products'],
                $material
            );

            // Generate progress ID
            $progressId = 'sale_price_progress_' . uniqid();

            // CRITICAL: Store the initial total in a separate key that will NEVER change
            $initialTotalKey = "{$progressId}_initial_total";
            Cache::put($initialTotalKey, $totalVariants, 3600); // Store for 1 hour

            // Initialize cache with total
            Cache::put($progressId, [
                'total' => $totalVariants,
                'processed' => 0,
                'errors' => 0,
                'status' => 'processing',
                'current_variant' => 'Initializing...',
            ], 3600); // 1 hour TTL

            // Dispatch job with all data
            try {
                $job = new UpdateCollectionSalePriceJob(
                    $request->all(),
                    $shopifyData['products'],
                    $shopifyData['title'],
                    $progressId
                );

                dispatch($job);
            } catch (\Throwable $dispatchError) {
                Log::error("Failed to dispatch sale price update job", [
                    'progress_id' => $progressId,
                    'error' => $dispatchError->getMessage(),
                    'trace' => $dispatchError->getTraceAsString(),
                ]);
                throw $dispatchError;
            }

            // Return 202 Accepted with progress ID
            return response()->json([
                'status' => 202,
                'message' => 'Processing started',
                'data' => [
                    'progress_id' => $progressId
                ],
            ], 202);
        } catch (\Throwable $th) {
            Log::error("Failed to start sale price update", [
                'error' => $th->getMessage(),
                'trace' => $th->getTraceAsString(),
            ]);

            return response()->json([
                'status' => 500,
                'message' => 'Failed to start update: ' . $th->getMessage(),
                'data' => null,
            ], 500);
        }
    }

    public function updatePrice(Request $request)
    {

        $result = (new UpdateShopifyCollectionSalePrice())($request);

        $response = [
            'status' => 200,
            'message' => 'Update collection result',
            'data' => $result,
        ];

        return response()->json($response, $response['status']);
    }

    /**
     * Get products from a collection
     */
    public function getCollectionProducts(Request $request)
    {
        $collectionId = $request->query('collection_id');

        if (!$collectionId) {
            return response()->json([
                'status' => 400,
                'message' => 'Collection ID is required',
                'data' => null,
            ], 400);
        }

        try {
            $action = new UpdateShopifyCollectionCostPrice();
            $shopifyData = $action->fetchCollectionData($collectionId);
            
            // Extract products with id and title only
            $products = array_map(function ($item) {
                return [
                    'id' => $item['node']['id'],
                    'title' => $item['node']['title']
                ];
            }, $shopifyData['products']);

            return response()->json([
                'status' => 200,
                'message' => 'Products retrieved successfully',
                'data' => $products,
            ], 200);
        } catch (\Throwable $th) {
            Log::error("Failed to get collection products", [
                'collection_id' => $collectionId,
                'error' => $th->getMessage()
            ]);

            return response()->json([
                'status' => 500,
                'message' => 'Failed to get products: ' . $th->getMessage(),
                'data' => null,
            ], 500);
        }
    }

    /**
     * Get progress for sale price update
     */
    public function getProgress(Request $request)
    {
        $progressId = $request->query('progress_id');

        if (!$progressId) {
            return response()->json([
                'status' => 400,
                'message' => 'Progress ID is required',
                'data' => null,
            ], 400);
        }

        // CRITICAL: Always read the initial total from the separate key and use it
        $initialTotalKey = "{$progressId}_initial_total";
        $initialTotal = Cache::get($initialTotalKey);

        // Use lock to ensure consistent read (prevents reading during write)
        $lock = Cache::lock("progress_read_{$progressId}", 5);

        try {
            if ($lock->get()) {
                $progress = Cache::get($progressId);
                $lock->release();
            } else {
                // If lock can't be acquired, try reading without lock (fallback)
                $progress = Cache::get($progressId);
            }
        } catch (\Throwable $lockError) {
            // Fallback: read without lock
            $progress = Cache::get($progressId);
        }

        if (!$progress) {
            return response()->json([
                'status' => 404,
                'message' => 'Progress not found',
                'data' => null,
            ], 404);
        }

        // Ensure progress has all required fields
        $progress = array_merge([
            'total' => 0,
            'processed' => 0,
            'errors' => 0,
            'status' => 'processing',
            'current_variant' => null,
        ], $progress);

        // CRITICAL: Override the total with the initial total if it exists
        if ($initialTotal !== null && $initialTotal > 0) {
            $progress['total'] = $initialTotal;
        }

        return response()->json([
            'status' => 200,
            'message' => 'Progress retrieved successfully',
            'data' => $progress,
        ], 200);
    }
}
