<?php

namespace App\Http\Controllers;

use App\Actions\ListShopifyCollectionsAction;
use App\Http\Requests\UpdateConfigurationRequest;
use App\Models\Configuration;
use App\Models\GoldPrice;
use App\Models\GoldPriceWeightRule;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Inertia\Inertia;

class ConfigurationController extends Controller
{
    public function index()
    {
        $configuration = Configuration::getInstance();

        // Get latest gold price
        $latestGoldPrice = GoldPrice::latest('created_at')->first();
        $currentGoldPrice = $latestGoldPrice ? $latestGoldPrice->price : 2000; // Default to 2000 if no price exists

        // Get weight rules grouped by collection_id
        $allWeightRules = GoldPriceWeightRule::ordered()->get();
        $weightRulesByCollection = $allWeightRules->groupBy('collection_id')->map(function ($rules) {
            return $rules->values();
        })->toArray();

        return Inertia::render('configuration', [
            'configuration' => $configuration,
            'currentGoldPrice' => $currentGoldPrice,
            'weightRulesByCollection' => $weightRulesByCollection
        ]);
    }

    public function update(UpdateConfigurationRequest $request)
    {
        $configuration = Configuration::getInstance();

        $collections = $request->input('collections', []);
        
        // For each collection, try to find and store its Shopify collection_id if not already present
        foreach ($collections as &$collection) {
            if (empty($collection['collection_id']) && !empty($collection['name'])) {
                try {
                    $collectionData = (new ListShopifyCollectionsAction())($collection['name']);
                    if (!empty($collectionData['collections'])) {
                        $shopifyCollection = $collectionData['collections'][0];
                        $collection['collection_id'] = is_array($shopifyCollection) ? $shopifyCollection['id'] : $shopifyCollection->id;
                    }
                } catch (\Throwable $th) {
                    Log::warning("Could not find collection_id for collection", [
                        'name' => $collection['name'],
                        'error' => $th->getMessage()
                    ]);
                }
            }
        }
        unset($collection); // Break reference

        $configuration->update([
            'auto_10k' => $request->input('auto_10k'),
            'auto_14k' => $request->input('auto_14k'),
            'collections' => $collections,
            'rounding_option' => $request->input('rounding_option', 'default'),
            'compare_price_config' => $request->input('compare_price_config'),
        ]);

        return redirect()->route('configuration.index')->with('success', 'Configuration updated successfully');
    }

    /**
     * Get all weight rules for a specific collection
     */
    public function getWeightRules(Request $request)
    {
        $collectionId = $request->query('collection_id');
        
        if ($collectionId) {
            $rules = GoldPriceWeightRule::forCollection($collectionId)->ordered()->get();
        } else {
            // For backward compatibility, return all rules if no collection_id specified
            $rules = GoldPriceWeightRule::ordered()->get();
        }
        
        return response()->json($rules);
    }

    /**
     * Store a new weight rule
     */
    public function storeWeightRule(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'collection_id' => ['required', 'string'],
            'min_weight' => ['required', 'numeric', 'min:0'],
            'max_weight' => ['required', 'numeric', 'min:0', 'gt:min_weight'],
            'fee_10k' => ['required', 'numeric', 'min:0'],
            'fee_14k' => ['required', 'numeric', 'min:0'],
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $collectionId = $request->input('collection_id');
        $minWeight = (float)$request->input('min_weight');
        $maxWeight = (float)$request->input('max_weight');

        // Check for overlapping ranges within the same collection
        $overlappingRules = GoldPriceWeightRule::forCollection($collectionId)
            ->where(function ($query) use ($minWeight, $maxWeight) {
                $query->where(function ($q) use ($minWeight, $maxWeight) {
                    // Check if new range overlaps with existing ranges
                    // Two ranges overlap if: min1 <= max2 && min2 <= max1
                    $q->where('min_weight', '<=', $maxWeight)
                      ->where('max_weight', '>=', $minWeight);
                });
            })->get();

        if ($overlappingRules->count() > 0) {
            return response()->json([
                'success' => false,
                'message' => 'This weight range overlaps with existing rules for this collection. Please adjust the range.',
                'overlapping_rules' => $overlappingRules->map(function ($rule) {
                    return [
                        'id' => $rule->id,
                        'range' => "{$rule->min_weight}g - {$rule->max_weight}g"
                    ];
                })
            ], 422);
        }

        // Get the highest order value for this collection and add 1
        $maxOrder = GoldPriceWeightRule::forCollection($collectionId)->max('order') ?? 0;
        
        $rule = GoldPriceWeightRule::create([
            'collection_id' => $collectionId,
            'min_weight' => $minWeight,
            'max_weight' => $maxWeight,
            'fee_10k' => (float)$request->input('fee_10k'),
            'fee_14k' => (float)$request->input('fee_14k'),
            'order' => $maxOrder + 1,
        ]);

        return response()->json([
            'success' => true,
            'rule' => $rule
        ]);
    }

    /**
     * Update a weight rule
     */
    public function updateWeightRule(Request $request, $id)
    {
        $rule = GoldPriceWeightRule::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'min_weight' => ['required', 'numeric', 'min:0'],
            'max_weight' => ['required', 'numeric', 'min:0', 'gt:min_weight'],
            'fee_10k' => ['required', 'numeric', 'min:0'],
            'fee_14k' => ['required', 'numeric', 'min:0'],
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        // Check for overlapping ranges within the same collection (excluding current rule)
        $minWeight = (float)$request->input('min_weight');
        $maxWeight = (float)$request->input('max_weight');
        $collectionId = $rule->collection_id; // Use the rule's existing collection_id

        $overlappingRules = GoldPriceWeightRule::where('id', '!=', $id)
            ->forCollection($collectionId)
            ->where(function ($query) use ($minWeight, $maxWeight) {
                $query->where(function ($q) use ($minWeight, $maxWeight) {
                    $q->where('min_weight', '<=', $maxWeight)
                      ->where('max_weight', '>=', $minWeight);
                });
            })->get();

        if ($overlappingRules->count() > 0) {
            return response()->json([
                'success' => false,
                'message' => 'This weight range overlaps with existing rules for this collection. Please adjust the range.',
                'overlapping_rules' => $overlappingRules->map(function ($r) {
                    return [
                        'id' => $r->id,
                        'range' => "{$r->min_weight}g - {$r->max_weight}g"
                    ];
                })
            ], 422);
        }

        $rule->update([
            'min_weight' => $minWeight,
            'max_weight' => $maxWeight,
            'fee_10k' => (float)$request->input('fee_10k'),
            'fee_14k' => (float)$request->input('fee_14k'),
        ]);

        return response()->json([
            'success' => true,
            'rule' => $rule
        ]);
    }

    /**
     * Delete a weight rule
     */
    public function deleteWeightRule($id)
    {
        $rule = GoldPriceWeightRule::findOrFail($id);
        $rule->delete();

        return response()->json([
            'success' => true,
            'message' => 'Weight rule deleted successfully'
        ]);
    }
}
