<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Shopify\Clients\Graphql;

class GetCollectionName extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'get:collection-name {collection_id : Collection ID (e.g. gid://shopify/Collection/332635537560)}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Get collection name from Shopify by ID';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $collectionId = $this->argument('collection_id');

        $this->info("Querying Shopify for collection: {$collectionId}");
        $this->newLine();

        $shop = env('SHOPIFY_DOMAIN');
        $token = env('SHOPIFY_ACCESS_TOKEN');

        if (!$shop || !$token) {
            $this->error('Shopify configuration missing. Please check SHOPIFY_DOMAIN and SHOPIFY_ACCESS_TOKEN in .env');
            return 1;
        }

        try {
            $client = new Graphql($shop, $token);

            $query = <<<GRAPHQL
                query getCollection(\$id: ID!) {
                    collection(id: \$id) {
                        id
                        title
                        handle
                        description
                    }
                }
            GRAPHQL;

            $variables = [
                'id' => $collectionId,
            ];

            $response = $client->query([
                'query' => $query,
                'variables' => $variables,
            ]);

            $data = json_decode($response->getBody(), true);

            if (isset($data['data']['collection'])) {
                $collection = $data['data']['collection'];

                $this->info('✓ Collection found!');
                $this->newLine();
                $this->line('Collection Details:');
                $this->line('─────────────────────');
                $this->line("ID:        {$collection['id']}");
                $this->line("Name:      {$collection['title']}");
                $this->line("Handle:    " . ($collection['handle'] ?? 'N/A'));

                if (!empty($collection['description'])) {
                    $this->line("Description: {$collection['description']}");
                }

                return 0;
            } elseif (isset($data['errors'])) {
                $this->error('✗ Error querying Shopify:');
                foreach ($data['errors'] as $error) {
                    $this->error("  - " . ($error['message'] ?? json_encode($error)));
                }
                return 1;
            } else {
                $this->error('✗ Collection not found or access denied');
                $this->line('Response:');
                $this->line(json_encode($data, JSON_PRETTY_PRINT));
                return 1;
            }
        } catch (\Exception $e) {
            $this->error('✗ Error: ' . $e->getMessage());
            $this->error('File: ' . $e->getFile());
            $this->error('Line: ' . $e->getLine());
            return 1;
        }
    }
}
