<?php

namespace App\Console\Commands;

use App\Actions\GetMaterialDataByCollectionAction;
use App\Actions\UpdateShopifyCollectionSalePriceFrontGold;
use App\Helpers\GoldPriceHelper;
use App\Models\Configuration;
use App\Models\GoldPrice;
use Illuminate\Console\Command;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Shopify\Clients\Graphql;

class DebugCollectionGoldPrice extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'gold:debug-collection {collection_id? : Shopify collection ID (e.g., gid://shopify/Collection/334502887576)} {--name= : Collection name to search} {--list : List all collections}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Debug gold price update process for a specific collection (TEMPORARY COMMAND)';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $listCollections = $this->option('list');
        $collectionName = $this->option('name');
        $collectionId = $this->argument('collection_id');
        
        $this->info("=== DEBUG: Gold Price Update for Collection ===");
        $this->newLine();

        // If --list option, show all collections
        if ($listCollections) {
            $this->listAllCollections();
            return 0;
        }

        // If name is provided, search by name first
        if ($collectionName && !$collectionId) {
            $this->info("Searching collection by name: {$collectionName}");
            $collectionId = $this->findCollectionByName($collectionName);
            if (!$collectionId) {
                $this->error("Collection not found by name: {$collectionName}");
                $this->line("Use --list to see all available collections");
                return 1;
            }
            $this->info("Found collection ID: {$collectionId}");
            $this->newLine();
        }

        if (!$collectionId) {
            $this->error("Please provide either collection_id or --name option");
            $this->line("Usage examples:");
            $this->line("  php artisan gold:debug-collection gid://shopify/Collection/334502887576");
            $this->line("  php artisan gold:debug-collection --name='Collection Name'");
            $this->line("  php artisan gold:debug-collection --list");
            return 1;
        }
        
        $this->info("Collection ID: {$collectionId}");
        $this->newLine();

        // Step 1: Verify Shopify connection
        $this->info("Step 1: Verifying Shopify connection...");
        $shop = env('SHOPIFY_DOMAIN');
        $token = env('SHOPIFY_ACCESS_TOKEN');

        if (!$shop || !$token) {
            $this->error("Shopify configuration missing!");
            return 1;
        }
        $this->info("✓ Shopify connection OK");
        $this->newLine();

        // Step 2: Get collection information
        $this->info("Step 2: Fetching collection information...");
        try {
            $client = new Graphql($shop, $token);
            
            $query = <<<'GRAPHQL'
            query getCollectionInfo($collection_id: ID!) {
                collection(id: $collection_id) {
                    id
                    title
                    handle
                    products(first: 5) {
                        edges {
                            node {
                                id
                                title
                                status
                                metafield(namespace: "custom", key: "metal_purity") {
                                    value
                                }
                                variants(first: 3) {
                                    edges {
                                        node {
                                            id
                                            title
                                            price
                                            materialMetafield: metafield(namespace: "custom", key: "material") {
                                                value
                                            }
                                            metalPurityMetafield: metafield(namespace: "custom", key: "metal_purity") {
                                                value
                                            }
                                            inventoryItem {
                                                measurement {
                                                    weight {
                                                        unit
                                                        value
                                                    }
                                                }
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
            GRAPHQL;

            $response = $client->query([
                "query" => $query,
                "variables" => ['collection_id' => $collectionId]
            ]);

            $data = json_decode($response->getBody(), true);
            
            if (isset($data['errors'])) {
                $this->error("GraphQL Errors:");
                foreach ($data['errors'] as $error) {
                    $this->error("  - " . $error['message']);
                    if (isset($error['extensions'])) {
                        $this->line("    Extensions: " . json_encode($error['extensions']));
                    }
                }
                $this->newLine();
                $this->line("Full response: " . json_encode($data, JSON_PRETTY_PRINT));
                return 1;
            }

            $collection = $data['data']['collection'] ?? null;
            
            if (!$collection) {
                $this->error("Collection not found!");
                $this->line("Response data: " . json_encode($data, JSON_PRETTY_PRINT));
                $this->newLine();
                $this->warn("Possible reasons:");
                $this->line("  1. Collection ID is incorrect");
                $this->line("  2. Collection doesn't exist");
                $this->line("  3. Collection is not accessible with current API token");
                return 1;
            }

            $collectionTitle = $collection['title'] ?? 'Unknown';
            $products = $collection['products']['edges'] ?? [];
            $productsCount = count($products);

            $this->info("✓ Collection found:");
            $this->line("  Title: {$collectionTitle}");
            $this->line("  ID: {$collectionId}");
            $this->line("  Total Products: {$productsCount}");
            $this->newLine();

        } catch (\Exception $e) {
            $this->error("Error fetching collection: " . $e->getMessage());
            return 1;
        }

        // Step 3: Check configuration
        $this->info("Step 3: Checking configuration...");
        $config = Configuration::getInstance();
        $auto10k = $config->auto_10k ?? false;
        $auto14k = $config->auto_14k ?? false;
        $collections = $config->collections ?? [];

        $this->info("Auto 10k: " . ($auto10k ? 'ENABLED' : 'DISABLED'));
        $this->info("Auto 14k: " . ($auto14k ? 'ENABLED' : 'DISABLED'));
        
        $this->line("Collections in config: " . count($collections));
        foreach ($collections as $idx => $collectionConfig) {
            $this->line("  Collection " . ($idx + 1) . ": " . ($collectionConfig['name'] ?? 'N/A') . " (ID: " . ($collectionConfig['collection_id'] ?? 'NOT SET') . ")");
        }
        $this->newLine();
        
        // Check if collection is in configuration
        $collectionInConfig = null;
        foreach ($collections as $collectionConfig) {
            $configCollectionId = $collectionConfig['collection_id'] ?? null;
            $configCollectionName = $collectionConfig['name'] ?? null;
            
            // Debug output
            $this->line("Comparing: Config ID='{$configCollectionId}' vs Search ID='{$collectionId}'");
            $this->line("Comparing: Config Name='{$configCollectionName}' vs Search Name='{$collectionTitle}'");
            
            if ($configCollectionId && $configCollectionId === $collectionId) {
                $collectionInConfig = $collectionConfig;
                $this->info("  → Match found by ID!");
                break;
            }
            // Also check by name (case-insensitive)
            if ($configCollectionName && strtolower(trim($configCollectionName)) === strtolower(trim($collectionTitle))) {
                $collectionInConfig = $collectionConfig;
                $this->info("  → Match found by name!");
                break;
            }
        }
        $this->newLine();

        if ($collectionInConfig) {
            $this->info("✓ Collection found in configuration:");
            $this->line("  Name: " . ($collectionInConfig['name'] ?? 'N/A'));
            $this->line("  Enabled: " . (($collectionInConfig['enabled'] ?? false) ? 'YES' : 'NO'));
            $this->line("  Collection ID in config: " . ($collectionInConfig['collection_id'] ?? 'NOT SET'));
        } else {
            $this->warn("⚠ Collection NOT found in configuration!");
            $this->line("  This collection will NOT be processed in automatic updates.");
        }
        $this->newLine();

        // Step 4: Get material data
        $this->info("Step 4: Analyzing material data...");
        try {
            $materialData = (new GetMaterialDataByCollectionAction())($collectionId);
            
            $this->info("Total products in collection: {$materialData->total}");
            $this->info("Has 10k products: " . ($materialData->count_10K ? 'YES' : 'NO'));
            $this->info("Has 14k products: " . ($materialData->count_14K ? 'YES' : 'NO'));
            
            if ($materialData->count_10K) {
                $this->line("  10k Price: $" . number_format($materialData->price_10K, 2));
                $this->line("  10k Weight: " . number_format($materialData->weight_10K, 3) . "g");
                $this->line("  10k Price per gram: $" . number_format($materialData->current_price_10K, 2));
            }
            
            if ($materialData->count_14K) {
                $this->line("  14k Price: $" . number_format($materialData->price_14K, 2));
                $this->line("  14k Weight: " . number_format($materialData->weight_14K, 3) . "g");
                $this->line("  14k Price per gram: $" . number_format($materialData->current_price_14K, 2));
            }
            $this->newLine();

        } catch (\Exception $e) {
            $this->error("Error getting material data: " . $e->getMessage());
            $this->line("Trace: " . $e->getTraceAsString());
            return 1;
        }

        // Step 5: Check gold price
        $this->info("Step 5: Checking gold price...");
        $latestGoldPrice = GoldPrice::latest('created_at')->first();
        
        if (!$latestGoldPrice) {
            $this->error("No gold price found in database!");
            return 1;
        }

        $this->info("Latest gold price: $" . number_format($latestGoldPrice->price, 2) . " per troy ounce");
        $this->info("Gold price per gram: $" . number_format($latestGoldPrice->price / 31.1035, 2));
        $this->newLine();

        // Step 6: Calculate expected prices
        $this->info("Step 6: Calculating expected prices...");
        if ($materialData->count_10K) {
            $pricePerGram10k = GoldPriceHelper::calculateGoldPricePerGram(
                $collectionTitle,
                '10k',
                $latestGoldPrice->price,
                null,
                $collectionId
            );
            $this->info("Expected 10k price per gram: $" . number_format($pricePerGram10k, 2));
        }
        
        if ($materialData->count_14K) {
            $pricePerGram14k = GoldPriceHelper::calculateGoldPricePerGram(
                $collectionTitle,
                '14k',
                $latestGoldPrice->price,
                null,
                $collectionId
            );
            $this->info("Expected 14k price per gram: $" . number_format($pricePerGram14k, 2));
        }
        $this->newLine();

        // Step 7: Simulate update (dry run)
        $this->info("Step 7: Simulating update process (DRY RUN - no actual updates)...");
        $this->warn("This will show what would happen but won't make actual changes.");
        $this->newLine();

        // Check if collection would be processed
        if (!$collectionInConfig || !($collectionInConfig['enabled'] ?? false)) {
            $this->error("❌ Collection would NOT be processed because:");
            if (!$collectionInConfig) {
                $this->line("  - Collection is not in configuration");
            } else {
                $this->line("  - Collection is disabled in configuration");
            }
            $this->newLine();
        } else {
            $this->info("✓ Collection would be processed");
            
            // Check if materials would be updated
            if ($auto10k && $materialData->count_10K) {
                $this->info("  → 10k products would be updated");
            } elseif ($auto10k && !$materialData->count_10K) {
                $this->warn("  → 10k auto-update is enabled but no 10k products found");
            } elseif (!$auto10k) {
                $this->line("  → 10k auto-update is disabled");
            }

            if ($auto14k && $materialData->count_14K) {
                $this->info("  → 14k products would be updated");
            } elseif ($auto14k && !$materialData->count_14K) {
                $this->warn("  → 14k auto-update is enabled but no 14k products found");
            } elseif (!$auto14k) {
                $this->line("  → 14k auto-update is disabled");
            }
            $this->newLine();
        }

        // Step 8: Show sample products
        $this->info("Step 8: Sample products (first 5):");
        if (empty($products)) {
            $this->warn("  No products found in collection!");
        } else {
            foreach ($products as $index => $productEdge) {
                $product = $productEdge['node'];
                $productTitle = $product['title'] ?? 'Unknown';
                $productStatus = $product['status'] ?? 'UNKNOWN';
                $metalPurity = $product['metafield']['value'] ?? 'N/A';
                $variants = $product['variants']['edges'] ?? [];
                
                $this->line("  Product " . ($index + 1) . ": {$productTitle}");
                $this->line("    Status: {$productStatus}");
                $this->line("    Metal Purity: {$metalPurity}");
                $this->line("    Variants: " . count($variants));
                
                foreach ($variants as $variantIndex => $variantEdge) {
                    $variant = $variantEdge['node'];
                    $variantTitle = $variant['title'] ?? 'Default';
                    $price = $variant['price'] ?? '0';
                    $material = $variant['metafield']['value'] ?? 'N/A';
                    $weight = $variant['inventoryItem']['measurement']['weight']['value'] ?? 0;
                    $weightUnit = $variant['inventoryItem']['measurement']['weight']['unit'] ?? 'g';
                    
                    $materialMetafield = $variant['materialMetafield']['value'] ?? 'N/A';
                    $metalPurityMetafield = $variant['metalPurityMetafield']['value'] ?? 'N/A';
                    
                    $this->line("      Variant " . ($variantIndex + 1) . ": {$variantTitle}");
                    $this->line("        Price: $" . number_format((float)$price, 2));
                    $this->line("        Material metafield (key: material): {$materialMetafield}");
                    $this->line("        Metal Purity metafield (key: metal_purity): {$metalPurityMetafield}");
                    $this->line("        Weight: {$weight} {$weightUnit}");
                }
                $this->newLine();
            }
        }

        // Summary
        $this->newLine();
        $this->info("=== SUMMARY ===");
        $this->line("Collection ID: {$collectionId}");
        $this->line("Collection Name: {$collectionTitle}");
        $this->line("Total Products: {$productsCount}");
        $this->line("In Configuration: " . ($collectionInConfig ? 'YES' : 'NO'));
        $this->line("Enabled in Config: " . (($collectionInConfig['enabled'] ?? false) ? 'YES' : 'NO'));
        $this->line("Has 10k Products: " . ($materialData->count_10K ? 'YES' : 'NO'));
        $this->line("Has 14k Products: " . ($materialData->count_14K ? 'YES' : 'NO'));
        $this->line("Auto 10k Enabled: " . ($auto10k ? 'YES' : 'NO'));
        $this->line("Auto 14k Enabled: " . ($auto14k ? 'YES' : 'NO'));

        $wouldProcess = $collectionInConfig && ($collectionInConfig['enabled'] ?? false);
        $wouldUpdate10k = $wouldProcess && $auto10k && $materialData->count_10K;
        $wouldUpdate14k = $wouldProcess && $auto14k && $materialData->count_14K;

        $this->newLine();
        if ($wouldUpdate10k || $wouldUpdate14k) {
            $this->info("✓ Collection WOULD be processed and updated");
        } else {
            $this->error("❌ Collection would NOT be processed/updated");
            if (!$wouldProcess) {
                $this->line("Reason: Collection not in config or disabled");
            } elseif (!$auto10k && !$auto14k) {
                $this->line("Reason: Auto-update disabled for both 10k and 14k");
            } elseif (!$materialData->count_10K && !$materialData->count_14K) {
                $this->line("Reason: No 10k or 14k products found");
            }
        }

        return 0;
    }

    /**
     * Find collection ID by name
     */
    private function findCollectionByName($collectionName)
    {
        try {
            $shop = env('SHOPIFY_DOMAIN');
            $token = env('SHOPIFY_ACCESS_TOKEN');
            $client = new Graphql($shop, $token);

            $query = <<<GRAPHQL
            query {
                collections(first: 1, query: "title:{$collectionName}") {
                    edges {
                        node {
                            id
                            title
                        }
                    }
                }
            }
            GRAPHQL;

            $response = $client->query($query);
            $data = json_decode($response->getBody(), true);
            
            if (isset($data['data']['collections']['edges'][0])) {
                return $data['data']['collections']['edges'][0]['node']['id'];
            }
            
            return null;
        } catch (\Exception $e) {
            $this->error("Error searching collection: " . $e->getMessage());
            return null;
        }
    }

    /**
     * List all collections
     */
    private function listAllCollections()
    {
        try {
            $shop = env('SHOPIFY_DOMAIN');
            $token = env('SHOPIFY_ACCESS_TOKEN');
            $client = new Graphql($shop, $token);

            $query = <<<'GRAPHQL'
            {
                collections(first: 200) {
                    edges {
                        node {
                            id
                            title
                        }
                    }
                }
            }
            GRAPHQL;

            $response = $client->query($query);
            $data = json_decode($response->getBody(), true);
            
            $collections = $data['data']['collections']['edges'] ?? [];
            
            $this->info("Available Collections:");
            $this->newLine();
            
            $tableData = [];
            foreach ($collections as $edge) {
                $collection = $edge['node'];
                $tableData[] = [
                    'ID' => $collection['id'],
                    'Title' => $collection['title']
                ];
            }
            
            $this->table(['ID', 'Title'], $tableData);
            
            // Also show collections in configuration
            $this->newLine();
            $this->info("Collections in Configuration:");
            $config = Configuration::getInstance();
            $configuredCollections = $config->collections ?? [];
            
            if (empty($configuredCollections)) {
                $this->warn("  No collections configured");
            } else {
                foreach ($configuredCollections as $collectionConfig) {
                    $name = $collectionConfig['name'] ?? 'N/A';
                    $enabled = ($collectionConfig['enabled'] ?? false) ? 'YES' : 'NO';
                    $id = $collectionConfig['collection_id'] ?? 'NOT SET';
                    $this->line("  - {$name} (Enabled: {$enabled}, ID: {$id})");
                }
            }
            
        } catch (\Exception $e) {
            $this->error("Error listing collections: " . $e->getMessage());
        }
    }
}
