<?php

namespace App\Actions;

use App\Helpers\GoldPriceHelper;
use App\Helpers\PriceRoundingHelper;
use App\Models\GoldPrice;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Shopify\Clients\Graphql;
use stdClass;

class UpdateShopifyVariantPriceActionByMaterial
{
    public function __invoke($material, $productId, $price, $collectionName = null, $useWeightRules = false, $collectionId = null, $roundingOption = 'default', $demo = false, $fromPricePage = false)
    {
        try {



            $shop = env('SHOPIFY_DOMAIN');
            $token = env('SHOPIFY_ACCESS_TOKEN');

            if (!$shop || !$token) {
                Log::error("Shopify domain or access token is not set in environment variables.");
                // Return array format instead of Response object for consistency
                return [
                    'status' => 500,
                    'message' => 'Shopify configuration missing',
                    'data' => null,
                ];
            }

            $client = new Graphql($shop, $token);

            $query = <<<'GRAPHQL'
            query getProductVariants($id: ID!) {
                product(id: $id) {
                    id
                    title
                    metafield(namespace: "custom", key: "metal_purity") {
                        key
                        value
                    }
                    variants(first: 200) {
                        edges {
                            node {
                                id
                                title
                                price
                                metafield(namespace: "custom", key: "metal_purity") {
                                    key
                                    value
                                }
                                inventoryItem {
                                    measurement {
                                        weight {
                                            unit
                                            value
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        GRAPHQL;


            $variables = [
                'id' =>  $productId,
            ];


            $response = $client->query(["query" => $query, "variables" => $variables]);

            $data = json_decode($response->getBody(), true);

            $productsShop  =  $data['data']['product']['variants']['edges'] ?? [];

            $count_variants_updated = 0; // Count only variants of the specified material that were updated
            $count_variants_matching_material = 0; // Count variants that match the material
            $count_variants_error = 0;
            $product_updated = false; // Track if at least one variant of this product was updated
            $demoResults = []; // Store demo results
            $productTitle = $data['data']['product']['title'] ?? 'Unknown Product';

            foreach ($productsShop as $item) {

                try {
                    $variantId = $item['node']['id'];
                    $weight = $item['node']['inventoryItem']['measurement']['weight']['value'] ?? 0;

                    // Safely access metafield - it can be null
                    $metafield = $item['node']['metafield'] ?? null;
                    $metalPurity = '';
                    if ($metafield && isset($metafield['value'])) {
                        $metalPurity = strtolower($metafield['value']);
                    }

                    // Check if variant matches the material
                    if (!empty($metalPurity) && strpos($metalPurity, strtolower($material)) !== false) {
                        $count_variants_matching_material++;

                        // Calculate price: if useWeightRules is true, calculate per gram using weight rules
                        if ($useWeightRules && $collectionName && $weight > 0) {
                            $latestGoldPrice = GoldPrice::latest('created_at')->first();
                            if ($latestGoldPrice) {
                                $pricePerGram = GoldPriceHelper::calculateGoldPricePerGram(
                                    $collectionName,
                                    $material,
                                    $latestGoldPrice->price,
                                    $weight, // Pass weight to use weight-based rules
                                    $collectionId // Pass collection ID for collection-specific rules
                                );
                                $updatePrice = $pricePerGram * $weight;
                            } else {
                                // Fallback to fixed price if no gold price available
                                $updatePrice = $price * $weight;
                            }
                        } else {
                            // Use fixed price per gram (original behavior)
                            $updatePrice = $price * $weight;
                        }

                        // Apply rounding option (skipHundredRound = true only when called from /price page)
                        $updatePrice = PriceRoundingHelper::applyRounding($updatePrice, $roundingOption, $fromPricePage);

                        $currentPrice = (float)($item['node']['price'] ?? 0);

                        // Calculate price per gram for display
                        $pricePerGram = $price; // Default to the base price per gram
                        if ($useWeightRules && $weight > 0 && $collectionName) {
                            // If using weight rules, calculate the actual price per gram used
                            $latestGoldPrice = GoldPrice::latest('created_at')->first();
                            if ($latestGoldPrice) {
                                $pricePerGram = GoldPriceHelper::calculateGoldPricePerGram(
                                    $collectionName,
                                    $material,
                                    $latestGoldPrice->price,
                                    $weight,
                                    $collectionId
                                );
                            }
                        }

                        // In demo mode, collect results without updating
                        if ($demo) {
                            $demoResultItem = [
                                'product_title' => $productTitle,
                                'variant_title' => $item['node']['title'] ?? 'Unknown Variant',
                                'material' => strtoupper($material),
                                'weight' => (float)$weight,
                                'price_per_gram' => (float)$pricePerGram,
                                'before_price' => $currentPrice,
                                'after_price' => $updatePrice,
                            ];
                            $demoResults[] = $demoResultItem;
                            $count_variants_updated++;
                            $product_updated = true;
                        } else {
                            // Real mode: update prices
                            if ($updatePrice != $currentPrice) {
                                $internal_result = (new UpdateShopifyVariantPriceAction())($productId, $variantId, $updatePrice);
                                if ($internal_result) {
                                    $count_variants_updated++;
                                    $product_updated = true; // Product has at least one variant updated
                                } else {
                                    $count_variants_error++;
                                    Log::error("Error actualizando precio");
                                }
                            } else {
                                // Price is already the same, count it as updated (no change needed)
                                $count_variants_updated++;
                                $product_updated = true; // Product has at least one variant updated
                            }
                        }
                    }

                    //break;
                } catch (\Throwable $th) {
                    Log::error("Error actualizando variante: " . $th->getMessage());
                    $count_variants_error++;
                }
            }

            // Use array instead of stdClass for better JSON encoding
            $result = [
                'variants_updated' => $count_variants_updated, // Real count of updated variants matching material
                'variants_matching_material' => $count_variants_matching_material, // Total variants matching material
                'variant_error' => $count_variants_error,
                'product_updated' => $product_updated ? 1 : 0, // 1 if product was updated (has at least one variant updated), 0 otherwise
            ];

            if ($demo) {
                $result['demo_results'] = $demoResults; // Include demo results
            }

            $response = [
                'status' => 200,
                'message' => 'ok',
                'data' => $result,
            ];

            return $response;
        } catch (\Throwable $th) {

            /*Log::error("Failed to update Shopify products variants", [
                'message' => $th->getMessage(),
                'trace' => $th->getTraceAsString(),
            ]);*/

            $response = [
                'status' => 500,
                'message' => $th->getMessage(),
                'data' => null,
            ];

            return $response;
        }
    }
}
