<?php

namespace App\Actions;

use App\Helpers\GoldPriceHelper;
use App\Helpers\PriceRoundingHelper;
use App\Jobs\SendGoldPriceUpdateEmail;
use App\Models\GoldPrice;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;

use Shopify\Clients\Graphql;
use stdClass;

class UpdateShopifyCollectionSalePriceFrontGold
{
    public function __invoke(Request $request, $sendEmail = true)
    {
        try {
            $operation = $request->input('operation');

            $collection_id = $request->input('collection_id');
            $collection_name = $request->input('collection_name') ?? $request->input('_collection_title');
            $material = $request->input('material');
            $priceManual = $request->input('price'); // Get manual price if sent from frontend
            $roundingOption = $request->input('rounding_option', 'default');
            $onlyDraft = $request->input('only_draft', false);
            $demo = $request->input('demo', false);
            $progressIdFromRequest = $request->input('_progress_id');
            // Detect if this is called from /price page (manual update) vs automatic job
            // The controller sets _from_price_page = true for /price requests
            $fromPricePage = $request->input('_from_price_page', false);

            // Determine which price to use
            if ($priceManual !== null && $priceManual !== '') {
                // Use manual price from frontend (button "Update")
                $price = round(floatval($priceManual));
            } else {
                // Calculate price using helper (button "Use Gold Price")
                $latestGoldPrice = GoldPrice::latest('created_at')->first();

                if (!$latestGoldPrice) {
                    Log::error('No gold price found in database');
                    throw new \RuntimeException("No gold price found in database");
                }

                $pricePerGram = GoldPriceHelper::calculateGoldPricePerGram($collection_name, $material, $latestGoldPrice->price);
                $price = round($pricePerGram); // Round to integer
            }

            // Use pre-obtained products data if available (from Job), otherwise fetch from Shopify
            $productsShop = $request->input('_products_data');

            if (!$productsShop) {
                // Fetch from Shopify if not provided
                $shop = env('SHOPIFY_DOMAIN');
                $token = env('SHOPIFY_ACCESS_TOKEN');

                if (!$shop || !$token) {
                    Log::error("Shopify domain or access token is not set in environment variables.");
                    return response()->json(['error' => 'Shopify configuration missing'], 500);
                }

                $client = new Graphql($shop, $token);

                $query = <<<'GRAPHQL'
                query getProductsFromCollection($collection_id: ID!) {
                    collection(id: $collection_id) {
                        id
                        title
                        products(first: 200) {
                            edges {
                                node {
                                    id
                                    title
                                    status
                                    variants(first: 200) {
                                        edges {
                                            node {
                                                id
                                                title
                                                price
                                                compareAtPrice
                                                metafield(namespace: "custom", key: "material") {
                                                    key
                                                    value
                                                }
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            GRAPHQL;

                $variables = [
                    'collection_id' =>  $collection_id,
                ];

                $response = $client->query(["query" => $query, "variables" => $variables]);
                $data = json_decode($response->getBody(), true);
                $productsShop = $data['data']['collection']['products']['edges'] ?? [];
            }

            // Filter products if specific products are selected
            $selectedProductIds = $request->input('selected_product_ids');
            if ($selectedProductIds && is_array($selectedProductIds) && count($selectedProductIds) > 0) {
                $beforeCount = count($productsShop);
                $productsShop = array_filter($productsShop, function ($item) use ($selectedProductIds) {
                    return in_array($item['node']['id'], $selectedProductIds);
                });
                // Re-index array after filtering
                $productsShop = array_values($productsShop);
            }

            // Filter products by status (only draft) if only_draft is true
            if ($onlyDraft) {
                $productsShop = array_filter($productsShop, function ($item) {
                    $status = strtoupper($item['node']['status'] ?? '');
                    return $status === 'DRAFT';
                });
                $productsShop = array_values($productsShop); // Re-index array
            }

            // Use progress ID from request if provided (for early initialization), otherwise generate one
            $progressId = $progressIdFromRequest ?? 'sale_price_progress_' . uniqid();

            // Pre-calculate total variants matching the material (needed for accurate progress tracking)
            $FINAL_TOTAL_VARIANTS = $this->calculateTotalVariants($productsShop, $material);

            // If we have a progress_id from controller, ALWAYS use the initial total stored separately
            if ($progressIdFromRequest) {
                // Read the initial total from the separate key FIRST
                $initialTotalKey = "{$progressIdFromRequest}_initial_total";
                $initialTotal = Cache::get($initialTotalKey);
                $finalTotal = ($initialTotal !== null && $initialTotal > 0) ? $initialTotal : $FINAL_TOTAL_VARIANTS;

                $lock = Cache::lock("progress_{$progressIdFromRequest}", 10);
                try {
                    if ($lock->get()) {
                        $cachedProgress = Cache::get($progressIdFromRequest, []);
                        Cache::put($progressIdFromRequest, array_merge($cachedProgress, [
                            'total' => $finalTotal,
                            'status' => $cachedProgress['status'] ?? 'processing',
                        ]), 3600);
                        $lock->release();
                    } else {
                        $cachedProgress = Cache::get($progressIdFromRequest, []);
                        Cache::put($progressIdFromRequest, array_merge($cachedProgress, [
                            'total' => $finalTotal,
                            'status' => $cachedProgress['status'] ?? 'processing',
                        ]), 3600);
                    }
                } catch (\Throwable $lockError) {
                    $finalTotal = ($initialTotal !== null && $initialTotal > 0) ? $initialTotal : $FINAL_TOTAL_VARIANTS;
                    $cachedProgress = Cache::get($progressIdFromRequest, []);
                    Cache::put($progressIdFromRequest, array_merge($cachedProgress, [
                        'total' => $finalTotal,
                        'status' => $cachedProgress['status'] ?? 'processing',
                    ]), 3600);
                }
            }

            // Initialize progress in cache (only if not already initialized by controller)
            if (!$progressIdFromRequest) {
                Cache::put($progressId, [
                    'total' => $FINAL_TOTAL_VARIANTS,
                    'processed' => 0,
                    'errors' => 0,
                    'status' => 'processing',
                    'current_variant' => 'Initializing...',
                ], 3600);
            }

            $totalVariantsUpdated = 0; // Total variants of the specified material that were updated
            $totalVariantsProcessed = 0; // Total variants of the specified material that were processed
            $totalVariantsErrors = 0;
            $totalProductsUpdated = 0; // Total products updated (products with at least one variant updated)
            $demoResults = []; // Store results for demo mode
            $processedCount = 0; // Counter for progress updates
            $progressUpdateInterval = 10; // Update cache every 10 variants

            // Get latest gold price for weight-based rules (if not using manual price)
            $latestGoldPrice = null;
            $useWeightRules = false;
            if ($priceManual === null || $priceManual === '') {
                $latestGoldPrice = GoldPrice::latest('created_at')->first();
                $useWeightRules = true; // Use weight-based rules when calculating from gold price
            }

            foreach ($productsShop as $items) {
                $productId = $items['node']['id'];
                $productTitle = $items['node']['title'] ?? 'Unknown Product';
                try {
                    $result = (new UpdateShopifyVariantPriceActionByMaterial())(
                        $material,
                        $productId,
                        $price,
                        $collection_name,
                        $useWeightRules,
                        $collection_id, // Pass collection ID for collection-specific rules
                        $roundingOption, // Pass rounding option
                        $demo, // Pass demo mode flag
                        $fromPricePage // Pass flag to indicate if called from /price page
                    );

                    // Extract variant count from result (handle both array and object formats)
                    if ($result && is_array($result) && isset($result['data'])) {
                        $variantData = $result['data'];

                        // Convert object to array if needed
                        if (is_object($variantData)) {
                            $variantData = json_decode(json_encode($variantData), true);
                        }

                        // Now handle as array
                        if (is_array($variantData)) {
                            if (isset($variantData['variants_updated'])) {
                                $totalVariantsUpdated += (int)$variantData['variants_updated'];
                            }
                            if (isset($variantData['variants_matching_material'])) {
                                $variantsMatching = (int)$variantData['variants_matching_material'];
                                $totalVariantsProcessed += $variantsMatching;
                                $processedCount += $variantsMatching;
                            }
                            if (isset($variantData['variant_error'])) {
                                $totalVariantsErrors += (int)$variantData['variant_error'];
                            }
                            if (isset($variantData['product_updated']) && $variantData['product_updated']) {
                                $totalProductsUpdated++; // Count product as updated if it has at least one variant updated
                            }

                            // Update progress cache periodically (both demo and non-demo mode)
                            if ($processedCount % $progressUpdateInterval === 0 || $processedCount === 1) {
                                $initialTotalKey = "{$progressId}_initial_total";
                                $initialTotal = Cache::get($initialTotalKey);
                                $finalTotal = ($initialTotal !== null && $initialTotal > 0) ? $initialTotal : $FINAL_TOTAL_VARIANTS;

                                $lock = Cache::lock("progress_{$progressId}", 10);
                                try {
                                    if ($lock->get()) {
                                        Cache::put($progressId, [
                                            'total' => $finalTotal,
                                            'processed' => $processedCount,
                                            'errors' => $totalVariantsErrors,
                                            'status' => 'processing',
                                            'current_variant' => $productTitle ?? 'Processing...',
                                        ], 3600);
                                        $lock->release();
                                    } else {
                                        Cache::put($progressId, [
                                            'total' => $finalTotal,
                                            'processed' => $processedCount,
                                            'errors' => $totalVariantsErrors,
                                            'status' => 'processing',
                                            'current_variant' => $productTitle ?? 'Processing...',
                                        ], 3600);
                                    }
                                } catch (\Throwable $lockError) {
                                    Cache::put($progressId, [
                                        'total' => $finalTotal,
                                        'processed' => $processedCount,
                                        'errors' => $totalVariantsErrors,
                                        'status' => 'processing',
                                        'current_variant' => $productTitle ?? 'Processing...',
                                    ], 3600);
                                }
                            }

                            // Collect demo results
                            if ($demo && isset($variantData['demo_results'])) {
                                $productDemoResults = $variantData['demo_results'];
                                if (is_array($productDemoResults)) {
                                    $demoResults = array_merge($demoResults, $productDemoResults);
                                }
                            }
                        }
                    }
                } catch (\Exception $e) {
                    Log::error("Error updating variants for product", [
                        'product_id' => $productId,
                        'error' => $e->getMessage()
                    ]);
                    // Continue with next product instead of stopping the whole process
                }
            }

            // Mark progress as completed (both demo and non-demo mode)
            $initialTotalKey = "{$progressId}_initial_total";
            $initialTotal = Cache::get($initialTotalKey);
            $finalTotal = ($initialTotal !== null && $initialTotal > 0) ? $initialTotal : $FINAL_TOTAL_VARIANTS;

            $lock = Cache::lock("progress_{$progressId}", 10);
            try {
                if ($lock->get()) {
                    $progressData = [
                        'total' => $finalTotal,
                        'processed' => $processedCount,
                        'errors' => $totalVariantsErrors,
                        'status' => 'completed',
                        'current_variant' => null,
                    ];

                    // Include demo results if in demo mode
                    if ($demo && !empty($demoResults)) {
                        $progressData['demo_results'] = $demoResults;
                        $progressData['demo_mode'] = true;
                    }

                    Cache::put($progressId, $progressData, 3600);
                    $lock->release();
                } else {
                    $progressData = [
                        'total' => $finalTotal,
                        'processed' => $processedCount,
                        'errors' => $totalVariantsErrors,
                        'status' => 'completed',
                        'current_variant' => null,
                    ];

                    // Include demo results if in demo mode
                    if ($demo && !empty($demoResults)) {
                        $progressData['demo_results'] = $demoResults;
                        $progressData['demo_mode'] = true;
                    }

                    Cache::put($progressId, $progressData, 3600);
                }
            } catch (\Throwable $lockError) {
                $progressData = [
                    'total' => $finalTotal,
                    'processed' => $processedCount,
                    'errors' => $totalVariantsErrors,
                    'status' => 'completed',
                    'current_variant' => null,
                ];

                // Include demo results if in demo mode
                if ($demo && !empty($demoResults)) {
                    $progressData['demo_results'] = $demoResults;
                    $progressData['demo_mode'] = true;
                }

                Cache::put($progressId, $progressData, 3600);
            }

            // Send email notification only if sendEmail is true (manual updates from frontend) and not in demo mode
            if ($sendEmail && !$demo) {
                $latestGoldPrice = GoldPrice::latest('created_at')->first();
                $priceUsed = $priceManual !== null && $priceManual !== ''
                    ? $price
                    : GoldPriceHelper::calculateGoldPricePerGram($collection_name, $material, $latestGoldPrice->price);

                $emailCollections = [[
                    'name' => $collection_name,
                    'products_processed' => count($productsShop), // Total products in collection
                    'price_10k' => $material === '10k' ? $priceUsed : null,
                    'price_14k' => $material === '14k' ? $priceUsed : null,
                    'products_updated_10k' => $material === '10k' ? $totalProductsUpdated : 0, // Products updated (not variants)
                    'products_updated_14k' => $material === '14k' ? $totalProductsUpdated : 0, // Products updated (not variants)
                    'errors_10k' => $material === '10k' ? $totalVariantsErrors : 0,
                    'errors_14k' => $material === '14k' ? $totalVariantsErrors : 0,
                    'has_errors' => $totalVariantsErrors > 0,
                    'error_details' => [],
                ]];

                try {
                    SendGoldPriceUpdateEmail::dispatch('manual', $emailCollections, $latestGoldPrice->price);
                } catch (\Exception $e) {
                    Log::error("Failed to dispatch email notification", [
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString()
                    ]);
                }
            }

            $response = [
                'status' => 200,
                'message' => $demo ? 'Simulation completed successfully' : 'Prices updated successfully',
                'data' => [
                    'collection' => $collection_name,
                    'material' => $material,
                    'price' => $price,
                    'variants_updated' => $totalVariantsUpdated, // Real count of variants updated
                    'variants_processed' => $totalVariantsProcessed, // Real count of variants processed (matching material)
                    'products_updated' => $totalProductsUpdated, // Real count of products updated (products with at least one variant updated)
                    'variants_errors' => $totalVariantsErrors,
                    'demo_mode' => $demo,
                    'results' => $demo ? $demoResults : null, // Include results only in demo mode
                ],
            ];

            return response()->json($response, 200);
        } catch (\Throwable $th) {

            Log::error("Failed to update Shopify products variants", [
                'message' => $th->getMessage(),
                'trace' => $th->getTraceAsString(),
            ]);

            $response = [
                'status' => 500,
                'message' => $th->getMessage(),
                'data' => null,
            ];

            return response()->json($response, 500);
        }
    }

    /**
     * Calculate total variants matching the specified material
     */
    public function calculateTotalVariants(array $productsShop, string $material): int
    {
        $total = 0;
        $materialLower = strtolower($material);

        foreach ($productsShop as $item) {
            $variants = $item['node']['variants']['edges'] ?? [];

            foreach ($variants as $variant) {
                $metafield = $variant['node']['metafield'] ?? null;
                $variantMaterial = '';
                if ($metafield && isset($metafield['value'])) {
                    $variantMaterial = strtolower($metafield['value']);
                }

                // Check if variant material matches (10k, 14k, etc.)
                if (strpos($variantMaterial, $materialLower) !== false) {
                    $total++;
                }
            }
        }

        return $total;
    }
}
