<?php

namespace App\Actions;

use App\Helpers\PriceRoundingHelper;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

use Shopify\Clients\Graphql;
use stdClass;

class UpdateShopifyCollectionReferencePriceFromSale
{
    public function __invoke(Request $request)
    {
        try {
            // Validación de entrada
            $validated = $request->validate([
                'amount' => 'required|numeric|min:0',
                'calculation_type' => 'required|in:fixed,percentage,discount_based',
                'collection_id' => 'required|string',
                'demo' => 'nullable|boolean',
                'only_draft' => 'nullable|boolean',
                'rounding_option' => 'nullable|in:default,round_to_99_99,round_to_hundreds,round_to_hundreds_minus_one'
            ]);

            // Additional validation for discount_based: percentage must be less than 100
            if ($validated['calculation_type'] === 'discount_based' && $validated['amount'] >= 100) {
                throw new \Exception('Discount percentage must be less than 100%');
            }

            $amount = (float)$validated['amount'];
            $calculationType = $validated['calculation_type'];
            $collection_id = $validated['collection_id'];
            $demo = $validated['demo'] ?? false;
            $onlyDraft = $validated['only_draft'] ?? false;
            $roundingOption = $validated['rounding_option'] ?? 'default';

            // Configuración Shopify
            $shop = env('SHOPIFY_DOMAIN');
            $token = env('SHOPIFY_ACCESS_TOKEN');

            if (!$shop || !$token) {
                throw new \Exception('Shopify configuration missing');
            }

            $client = new Graphql($shop, $token);

            // Query GraphQL para obtener productos y variantes con precio de venta
            $query = <<<'GRAPHQL'
        query getProductsFromCollection($collection_id: ID!) {
            collection(id: $collection_id) {
                id
                title
                products(first: 200) {
                    edges {
                        node {
                            id
                            title
                            status
                            variants(first: 200) {
                                edges {
                                    node {
                                        id
                                        title
                                        price
                                        compareAtPrice
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        GRAPHQL;

            $response = $client->query([
                "query" => $query,
                "variables" => ['collection_id' => $collection_id]
            ]);

            $data = json_decode($response->getBody(), true);
            $productsShop = $data['data']['collection']['products']['edges'] ?? [];

            // Filter products if specific products are selected
            $selectedProductIds = $request->input('selected_product_ids');
            if ($selectedProductIds && is_array($selectedProductIds) && count($selectedProductIds) > 0) {
                $productsShop = array_filter($productsShop, function ($item) use ($selectedProductIds) {
                    return in_array($item['node']['id'], $selectedProductIds);
                });
                // Re-index array after filtering
                $productsShop = array_values($productsShop);
            }

            // Filter products by status (only draft) if only_draft is true
            if ($onlyDraft) {
                $productsShop = array_filter($productsShop, function ($item) {
                    return strtoupper($item['node']['status'] ?? '') === 'DRAFT';
                });
                $productsShop = array_values($productsShop); // Re-index array
            }

            $demoResult = new stdClass();
            $demoResult->collection = $data['data']['collection']['title'] ?? 'Unknown Collection';
            $demoObjects = [];
            $stats = ['success' => 0, 'errors' => 0];

            foreach ($productsShop as $product) {
                $productId = $product['node']['id'] ?? null;
                $productTitle = $product['node']['title'] ?? 'Unknown Product';

                foreach ($product['node']['variants']['edges'] ?? [] as $variantEdge) {
                    $variant = $variantEdge['node'] ?? [];
                    $variantId = $variant['id'] ?? null;
                    $variantTitle = $variant['title'] ?? 'Default Title';

                    $salePrice = (float)($variant['price'] ?? 0);
                    $originalComparePrice = (float)($variant['compareAtPrice'] ?? 0);

                    // Skip variants with no sale price
                    if ($salePrice <= 0) {
                        continue;
                    }

                    $demoObject = new stdClass();
                    $demoObject->product_title = $productTitle;
                    $demoObject->variant_title = $variantTitle;
                    $demoObject->before_price = $salePrice;
                    $demoObject->before_compare_price = $originalComparePrice;
                    $demoObject->after_compare_price = null;
                    $demoObject->error = null;

                    try {
                        // Calcular el nuevo compare price según el tipo
                        if ($calculationType === 'percentage') {
                            // Fórmula: Compare Price = Sale Price + (Sale Price × Percentage / 100)
                            $percentageAmount = $salePrice * ($amount / 100);
                            $newComparePrice = $salePrice + $percentageAmount;
                        } elseif ($calculationType === 'discount_based') {
                            // Fórmula: Compare Price = Sale Price / ((100 - Discount %) / 100)
                            // Ejemplo: 20% discount = Sale Price / 0.8
                            $discountFactor = (100 - $amount) / 100;
                            if ($discountFactor <= 0) {
                                throw new \Exception("Invalid discount percentage: discount factor must be greater than 0");
                            }
                            $newComparePrice = $salePrice / $discountFactor;
                        } else {
                            // Fórmula: Compare Price = Sale Price + Fixed Amount
                            $newComparePrice = $salePrice + $amount;
                        }

                        // Apply rounding option using helper
                        $newComparePrice = PriceRoundingHelper::applyRounding($newComparePrice, $roundingOption);

                        $demoObject->after_compare_price = $newComparePrice;

                        if (!$demo && $variantId) {
                            (new UpdateShopifyVariantPriceReferenceAction())(
                                $productId,
                                $variantId,
                                $newComparePrice
                            );
                        }

                        $stats['success']++;
                    } catch (\Throwable $e) {
                        $demoObject->error = $e->getMessage();
                        $stats['errors']++;
                        Log::error("Failed to update variant from sale price", [
                            'product' => $productTitle,
                            'variant' => $variantTitle,
                            'error' => $e->getMessage()
                        ]);
                    }

                    // Always add the demo object to the array, even if there was an error
                    // This ensures the table shows all results in test mode
                    $demoObjects[] = $demoObject;
                }
            }

            $demoResult->data = $demoObjects;
            $demoResult->stats = $stats;

            return $demoResult;
        } catch (\Throwable $th) {
            Log::error("Failed to update collection prices from sale price", [
                'error' => $th->getMessage(),
                'trace' => $th->getTraceAsString()
            ]);

            // Re-throw the exception so the controller can handle it properly
            throw $th;
        }
    }
}
