<?php

namespace App\Actions;

use App\Helpers\PriceRoundingHelper;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

use Shopify\Clients\Graphql;
use stdClass;

class UpdateShopifyCollectionReferencePrice
{
    public function __invoke(Request $request)
    {
        try {
            // Validación de entrada
            $validated = $request->validate([
                'operation' => 'required|in:Percent,Value',
                'action' => 'required|in:Increment,Decrement',
                'price' => 'required|numeric|min:0',
                'collection_id' => 'required|string',
                'demo' => 'nullable|boolean',
                'only_draft' => 'nullable|boolean',
                'rounding_option' => 'nullable|in:default,round_to_99_99,round_to_hundreds,round_to_hundreds_minus_one'
            ]);

            $operation = $validated['operation'];
            $action = $validated['action'];
            $price = (float)$validated['price'];
            $collection_id = $validated['collection_id'];
            $demo = $validated['demo'] ?? false;
            $onlyDraft = $validated['only_draft'] ?? false;
            $roundingOption = $validated['rounding_option'] ?? 'default';

            // Configuración Shopify
            $shop = env('SHOPIFY_DOMAIN');
            $token = env('SHOPIFY_ACCESS_TOKEN');

            if (!$shop || !$token) {
                throw new \Exception('Shopify configuration missing');
            }

            $client = new Graphql($shop, $token);

            // Query GraphQL
            $query = <<<'GRAPHQL'
        query getProductsFromCollection($collection_id: ID!) {
            collection(id: $collection_id) {
                id
                title
                products(first: 200) {
                    edges {
                        node {
                            id
                            title
                            status
                            variants(first: 200) {
                                edges {
                                    node {
                                        id
                                        title
                                        price
                                        compareAtPrice
                                        metafield(namespace: "custom", key: "material") {
                                            key
                                            value
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        GRAPHQL;

            $response = $client->query([
                "query" => $query,
                "variables" => ['collection_id' => $collection_id]
            ]);

            $data = json_decode($response->getBody(), true);
            $productsShop = $data['data']['collection']['products']['edges'] ?? [];

            // Filter products if specific products are selected
            $selectedProductIds = $request->input('selected_product_ids');
            if ($selectedProductIds && is_array($selectedProductIds) && count($selectedProductIds) > 0) {
                $productsShop = array_filter($productsShop, function ($item) use ($selectedProductIds) {
                    return in_array($item['node']['id'], $selectedProductIds);
                });
                // Re-index array after filtering
                $productsShop = array_values($productsShop);
            }

            // Filter products by status (only draft) if only_draft is true
            if ($onlyDraft) {
                $productsShop = array_filter($productsShop, function ($item) {
                    return strtoupper($item['node']['status'] ?? '') === 'DRAFT';
                });
                $productsShop = array_values($productsShop); // Re-index array
            }

            $demoResult = new stdClass();
            $demoResult->collection = $data['data']['collection']['title'] ?? 'Unknown Collection';
            $demoObjects = [];
            $stats = ['success' => 0, 'errors' => 0];

            foreach ($productsShop as $product) {
                $productId = $product['node']['id'] ?? null;
                $productTitle = $product['node']['title'] ?? 'Unknown Product';

                foreach ($product['node']['variants']['edges'] ?? [] as $variantEdge) {
                    $variant = $variantEdge['node'] ?? [];
                    $variantId = $variant['id'] ?? null;
                    $variantTitle = $variant['title'] ?? 'Default Title';

                    $originalPrice = (float)($variant['price'] ?? 0);
                    $originalComparePrice = (float)($variant['compareAtPrice'] ?? 0);

                    if ($originalPrice <= 0) {
                        continue;
                    }

                    $demoObject = new stdClass();
                    $demoObject->product_title = $productTitle;
                    $demoObject->variant_title = $variantTitle;
                    $demoObject->before_price = $originalPrice;
                    $demoObject->before_compare_price = $originalComparePrice;

                    try {
                        // Cálculo del incremento/decremento
                        $adjustment = ($operation === 'Percent')
                            ? $originalComparePrice * ($price / 100)
                            : $price;

                        $newComparePrice = ($action === "Increment")
                            ? $originalComparePrice + $adjustment
                            : max(0, $originalComparePrice - $adjustment); // Evitar precios negativos

                        // Apply rounding option using helper
                        $newComparePrice = PriceRoundingHelper::applyRounding($newComparePrice, $roundingOption);

                        $demoObject->after_compare_price = $newComparePrice;

                        if (!$demo && $variantId) {
                            (new UpdateShopifyVariantPriceReferenceAction())(
                                $productId,
                                $variantId,
                                $newComparePrice
                            );
                        }

                        $stats['success']++;
                        $demoObjects[] = $demoObject;
                    } catch (\Throwable $e) {
                        $demoObject->error = $e->getMessage();
                        $stats['errors']++;
                        Log::error("Failed to update variant", [
                            'product' => $productTitle,
                            'variant' => $variantTitle,
                            'error' => $e->getMessage()
                        ]);
                    }
                }
            }

            $demoResult->data = $demoObjects;
            $demoResult->stats = $stats;

            return $demoResult;
        } catch (\Throwable $th) {
            Log::error("Failed to update collection prices", [
                'error' => $th->getMessage(),
                'trace' => $th->getTraceAsString()
            ]);

            return response()->json([
                'status' => 500,
                'message' => 'Operation failed: ' . $th->getMessage(),
                'data' => null
            ], 500);
        }
    }
}
