<?php

namespace App\Actions;

use App\Models\CollectionCostConfiguration;
use App\Models\CollectionCostPriceLog;
use App\Helpers\PriceRoundingHelper;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;

use Shopify\Clients\Graphql;
use stdClass;

class UpdateShopifyCollectionCostPrice
{
    public function __invoke(Request $request)
    {
        try {
            $collection_id = $request->input('collection_id');
            $profit_ranges = $request->input('profit_ranges');
            $expenses = $request->input('expenses');
            $rounding_option = $request->input('rounding_option', 'default');
            $demo = $request->input('demo');
            $only_draft = $request->input('only_draft', false);

            // Get progress ID from request if provided (set by controller for early initialization)
            $progressIdFromRequest = $request->input('_progress_id');

            // Check if data is pre-obtained (from Job) or needs to be fetched
            $productsShop = $request->input('_products_data');
            $collectionTitle = $request->input('_collection_title');
            $excludedProductIds = $request->input('_excluded_product_ids', []);

            // Get selected product IDs from request (if any)
            $selectedProductIds = $request->input('selected_product_ids');

            // If data not pre-obtained, fetch from Shopify
            if (empty($productsShop)) {
                $shopifyData = $this->fetchCollectionData($collection_id);
                $productsShop = $shopifyData['products'];
                $collectionTitle = $shopifyData['title'];
                $excludedProductIds = $shopifyData['excluded_product_ids'];
            }

            // Filter products if specific products are selected
            if ($selectedProductIds && is_array($selectedProductIds) && count($selectedProductIds) > 0) {
                $productsShop = array_filter($productsShop, function ($item) use ($selectedProductIds) {
                    return in_array($item['node']['id'], $selectedProductIds);
                });
                // Re-index array after filtering
                $productsShop = array_values($productsShop);
            }

            // Filter products by status (only draft) if only_draft is true
            if ($only_draft) {
                $productsShop = array_filter($productsShop, function ($item) {
                    $status = $item['node']['status'] ?? null;
                    return strtoupper($status) === 'DRAFT';
                });
                // Re-index array after filtering
                $productsShop = array_values($productsShop);
            }

            $count_variants = 0;
            $count_variants_error = 0;
            $demoObjects = [];

            $demoResult = new stdClass();
            $demoResult->collection = $collectionTitle;

            // Use progress ID from request if provided (for early initialization), otherwise generate one
            $progressId = $progressIdFromRequest ?? 'cost_price_progress_' . uniqid();

            // Pre-calculate total variants (needed for accurate progress tracking)
            // ALWAYS calculate from actual data to ensure accuracy, don't rely on cache
            // The cache is only used for storing progress updates, not as source of truth for total
            // Store in a local constant-like variable that won't change during processing
            $FINAL_TOTAL_VARIANTS = $this->calculateTotalVariants($productsShop, $excludedProductIds);

            // Always use FINAL_TOTAL_VARIANTS for all cache updates
            $expectedTotalVariants = $FINAL_TOTAL_VARIANTS;

            // If we have a progress_id from controller, ALWAYS use the initial total stored separately
            // CRITICAL: The initial total is stored in a separate key and NEVER changes
            // This is the source of truth, regardless of any calculations
            if (!$demo && $progressIdFromRequest) {
                // Read the initial total from the separate key FIRST (before any lock)
                $initialTotalKey = "{$progressIdFromRequest}_initial_total";
                $initialTotal = Cache::get($initialTotalKey);

                $lock = Cache::lock("progress_{$progressIdFromRequest}", 10);

                try {
                    if ($lock->get()) {
                        // If initial total exists, use it. Otherwise fall back to calculated (shouldn't happen)
                        $finalTotal = ($initialTotal !== null && $initialTotal > 0) ? $initialTotal : $FINAL_TOTAL_VARIANTS;

                        $cachedProgress = Cache::get($progressIdFromRequest, []);

                        // Always update with the initial total (never changes)
                        Cache::put($progressIdFromRequest, array_merge($cachedProgress, [
                            'total' => $finalTotal, // Always use initial total from controller
                            'status' => $cachedProgress['status'] ?? 'processing',
                        ]), 3600);

                        $lock->release();
                    } else {
                        $finalTotal = ($initialTotal !== null && $initialTotal > 0) ? $initialTotal : $FINAL_TOTAL_VARIANTS;
                        $cachedProgress = Cache::get($progressIdFromRequest, []);
                        Cache::put($progressIdFromRequest, array_merge($cachedProgress, [
                            'total' => $finalTotal,
                            'status' => $cachedProgress['status'] ?? 'processing',
                        ]), 3600);
                    }
                } catch (\Throwable $lockError) {
                    // Fallback: read initial total from separate key
                    $finalTotal = ($initialTotal !== null && $initialTotal > 0) ? $initialTotal : $FINAL_TOTAL_VARIANTS;

                    $cachedProgress = Cache::get($progressIdFromRequest, []);
                    Cache::put($progressIdFromRequest, array_merge($cachedProgress, [
                        'total' => $finalTotal, // Always use initial total
                        'status' => $cachedProgress['status'] ?? 'processing',
                    ]), 3600);
                }
            }

            // Count total variants while processing (more efficient than separate loop)
            // But we'll use expectedTotalVariants for progress tracking
            $totalVariants = 0;
            $processedCount = 0;

            // Group updates by product to do bulk updates (reduces HTTP calls dramatically)
            $productUpdates = []; // [productId => ['price_updates' => [], 'compare_price_updates' => []]]

            // Progress update interval: update cache every N variants (Option 2)
            $progressUpdateInterval = 10; // Update every 10 variants

            // Initialize progress in cache (only if not already initialized by controller)
            // If progress ID was provided, assume cache was already initialized with total
            if (!$demo && !$progressIdFromRequest) {
                // Pre-calculate total before processing
                $totalVariantsToProcess = $this->calculateTotalVariants($productsShop, $excludedProductIds);

                Cache::put($progressId, [
                    'total' => $totalVariantsToProcess,
                    'processed' => 0,
                    'errors' => 0,
                    'status' => 'processing',
                    'current_variant' => 'Initializing...',
                ], 3600); // 1 hour TTL
            }

            foreach ($productsShop as $items) {
                $productId = $items['node']['id'];
                $productStatus = $items['node']['status'] ?? null;

                // Check if this product is in the excluded collection
                if (in_array($productId, $excludedProductIds)) {
                    continue; // Skip this product
                }

                // Additional check: if only_draft is enabled, skip non-draft products
                // (This is a safety check, as we already filtered above, but good to have)
                if ($only_draft && strtoupper($productStatus) !== 'DRAFT') {
                    continue; // Skip this product
                }

                $variants = $items['node']['variants']['edges'];

                // Initialize product updates array for this product
                if (!isset($productUpdates[$productId])) {
                    $productUpdates[$productId] = [
                        'price_updates' => [],
                        'compare_price_updates' => []
                    ];
                }

                foreach ($variants as $variant) {
                    // Get cost price from Shopify
                    $inventoryItem = $variant['node']['inventoryItem'] ?? null;
                    $unitCost = $inventoryItem['unitCost']['amount'] ?? null;
                    $costPrice = $unitCost ? (float)$unitCost : 0;

                    // Skip if no cost price
                    if ($costPrice <= 0) continue;

                    // Count total variants as we process (only once, not in separate loop)
                    $totalVariants++;

                    $demoObject = new stdClass();
                    $demoObject->id = $variant['node']['id'];
                    $demoObject->variant_title = $variant['node']['title'];
                    $demoObject->product_title = $items['node']['title'];

                    try {
                        $currentPrice = (float)($variant['node']['price'] ?? 0);
                        $comparePrice = (float)($variant['node']['compareAtPrice'] ?? 0);
                        $variantId = $variant['node']['id'] ?? null;

                        if (!$variantId) {
                            throw new \Exception("Missing variant ID");
                        }

                        $demoObject->before_sale_price = $currentPrice; // Current sale price (before update)
                        $demoObject->before_price = $costPrice; // Cost price
                        $demoObject->before_compare_price = $comparePrice;

                        // === APPLY COST CONFIGURATION FORMULA ===

                        // 1. Find profit for this cost
                        $profit = $this->getProfitForCost($costPrice, $profit_ranges);

                        // 2. Subtotal Base = Cost + Profit + Shipping
                        // Note: Extras will be added later if absolute, or calculated at the end if percentage
                        $subtotalBase = $costPrice + $profit + $expenses['shipping'];

                        // Add absolute extras to base if type is absolute
                        if (isset($expenses['extras_type']) && $expenses['extras_type'] === 'absolute') {
                            $subtotalBase += $expenses['extras'];
                        }

                        // 3. Commission (rounded if percentage)
                        if ($expenses['commission_type'] === 'percentage') {
                            $commission = round(($subtotalBase * $expenses['commission_value']) / 100);
                        } else {
                            $commission = (float)$expenses['commission_value'];
                        }

                        // 4. Subtotal before Gateway
                        $subtotalBeforeGateway = $subtotalBase + $commission;

                        // 5. Fix Gateway (always percentage, rounded)
                        $fixGateway = round(($subtotalBeforeGateway * $expenses['fix_gateway']) / 100);

                        // 6. Calculate Extras if percentage (on Cost + Profit + Fix Gateway Fee)
                        $extrasValue = 0;
                        if (isset($expenses['extras_type']) && $expenses['extras_type'] === 'percentage') {
                            // Extras percentage is calculated on: Cost + Profit + Fix Gateway Fee
                            $extrasBase = $costPrice + $profit + $fixGateway;
                            $extrasValue = round(($extrasBase * $expenses['extras']) / 100);
                        } else {
                            // Absolute extras already added to subtotalBase
                            $extrasValue = round($expenses['extras']);
                        }

                        // 7. Calculate base final price before rounding
                        $baseFinalPrice = $subtotalBeforeGateway + $fixGateway + $extrasValue;

                        // 8. Apply rounding option
                        $finalSalePrice = PriceRoundingHelper::applyRounding($baseFinalPrice, $rounding_option);

                        // === END FORMULA ===

                        $demoObject->after_sale_price = $finalSalePrice;

                        // IMPORTANT: Sale price should NEVER be greater than compare price
                        // If it is, we need to update the compare price too
                        if ($finalSalePrice > $comparePrice) {
                            $newComparePrice = round($finalSalePrice + 100);
                            $demoObject->after_compare_price = $newComparePrice;

                            if (!$demo) {
                                // Queue for bulk update instead of individual call
                                $productUpdates[$productId]['compare_price_updates'][] = [
                                    'id' => $variantId,
                                    'price' => $newComparePrice
                                ];
                            }
                        } else {
                            $demoObject->after_compare_price = $comparePrice;
                        }

                        // Queue for bulk update instead of individual call (if not in demo mode)
                        if (!$demo) {
                            $productUpdates[$productId]['price_updates'][] = [
                                'id' => $variantId,
                                'price' => $finalSalePrice
                            ];
                        }

                        // Add calculation breakdown for debugging (all values as integers)
                        $demoObject->calculation_breakdown = [
                            'cost' => round($costPrice),
                            'profit' => round($profit),
                            'shipping' => round($expenses['shipping']),
                            'extras_type' => $expenses['extras_type'] ?? 'absolute',
                            'extras_value' => round($expenses['extras']),
                            'extras' => round($extrasValue),
                            'subtotal_base' => round($subtotalBase),
                            'commission_type' => $expenses['commission_type'],
                            'commission_value' => round($expenses['commission_value']),
                            'commission' => round($commission),
                            'subtotal_before_gateway' => round($subtotalBeforeGateway),
                            'fix_gateway_percent' => $expenses['fix_gateway'],
                            'fix_gateway' => round($fixGateway),
                            'final_sale_price' => $finalSalePrice
                        ];

                        $count_variants++;
                        $demoObjects[] = $demoObject;
                        $processedCount++;

                        // Update progress in cache every N variants (Option 2: every 10 variants)
                        // Update cache every N variants OR on first variant to show immediate progress
                        if (!$demo && ($processedCount % $progressUpdateInterval === 0 || $processedCount === 1)) {
                            // CRITICAL: Always read the initial total from the separate key FIRST
                            // This total was set by the controller and NEVER changes
                            // IMPORTANT: Use progressIdFromRequest if available, otherwise use progressId
                            $actualProgressId = $progressIdFromRequest ?? $progressId;
                            $initialTotalKey = "{$actualProgressId}_initial_total";
                            $initialTotal = Cache::get($initialTotalKey);
                            $finalTotal = ($initialTotal !== null && $initialTotal > 0) ? $initialTotal : $FINAL_TOTAL_VARIANTS;

                            // Use lock to prevent race conditions
                            $lock = Cache::lock("progress_{$actualProgressId}", 10);

                            try {
                                if ($lock->get()) {
                                    Cache::put($actualProgressId, [
                                        'total' => $finalTotal, // Always use initial total from controller
                                        'processed' => $processedCount,
                                        'errors' => $count_variants_error,
                                        'status' => 'processing',
                                        'current_variant' => $demoObject->variant_title ?? 'Processing...',
                                    ], 3600);

                                    $lock->release();
                                } else {

                                    Cache::put($actualProgressId, [
                                        'total' => $finalTotal, // Always use initial total
                                        'processed' => $processedCount,
                                        'errors' => $count_variants_error,
                                        'status' => 'processing',
                                        'current_variant' => $demoObject->variant_title ?? 'Processing...',
                                    ], 3600);
                                }
                            } catch (\Throwable $lockError) {
                                // Fallback: read initial total from separate key and update
                                Cache::put($actualProgressId, [
                                    'total' => $finalTotal, // Always use initial total
                                    'processed' => $processedCount,
                                    'errors' => $count_variants_error,
                                    'status' => 'processing',
                                    'current_variant' => $demoObject->variant_title ?? 'Processing...',
                                ], 3600);
                            }
                        }
                    } catch (\Throwable $th) {
                        $demoObject->error = true;
                        $demoObject->error_message = $th->getMessage();
                        Log::error("Error updating price for variant: " . $th->getMessage());
                        $count_variants_error++;
                        $demoObjects[] = $demoObject;
                        $processedCount++;

                        // Update progress in cache every N variants (Option 2: every 10 variants)
                        // Update cache every N variants OR on first variant to show immediate progress
                        if (!$demo && ($processedCount % $progressUpdateInterval === 0 || $processedCount === 1)) {
                            // CRITICAL: Always read the initial total from the separate key FIRST
                            $initialTotalKey = "{$progressId}_initial_total";
                            $initialTotal = Cache::get($initialTotalKey);
                            $finalTotal = ($initialTotal !== null && $initialTotal > 0) ? $initialTotal : $FINAL_TOTAL_VARIANTS;

                            // Use lock to prevent race conditions
                            $lock = Cache::lock("progress_{$progressId}", 10);

                            try {
                                if ($lock->get()) {
                                    Cache::put($progressId, [
                                        'total' => $finalTotal, // Always use initial total from controller
                                        'processed' => $processedCount,
                                        'errors' => $count_variants_error,
                                        'status' => 'processing',
                                        'current_variant' => $demoObject->variant_title ?? 'Error occurred',
                                    ], 3600);

                                    $lock->release();
                                } else {
                                    Cache::put($progressId, [
                                        'total' => $finalTotal,
                                        'processed' => $processedCount,
                                        'errors' => $count_variants_error,
                                        'status' => 'processing',
                                        'current_variant' => $demoObject->variant_title ?? 'Error occurred',
                                    ], 3600);
                                }
                            } catch (\Throwable $lockError) {
                                Cache::put($progressId, [
                                    'total' => $finalTotal,
                                    'processed' => $processedCount,
                                    'errors' => $count_variants_error,
                                    'status' => 'processing',
                                    'current_variant' => $demoObject->variant_title ?? 'Error occurred',
                                ], 3600);
                            }
                        }
                    }
                }

                // Execute bulk updates for this product (after processing all variants)
                if (!$demo && isset($productUpdates[$productId])) {
                    $updateCount = count($productUpdates[$productId]['price_updates']) + count($productUpdates[$productId]['compare_price_updates']);
                    if ($updateCount > 0) {
                        $this->executeBulkUpdatesForProduct($productId, $productUpdates[$productId]);
                    }
                    // Clear after processing to free memory
                    unset($productUpdates[$productId]);
                }
            }

            // Mark progress as completed
            if (!$demo) {
                // CRITICAL: Always read the initial total from the separate key FIRST
                $initialTotalKey = "{$progressId}_initial_total";
                $initialTotal = Cache::get($initialTotalKey);
                $finalTotal = ($initialTotal !== null && $initialTotal > 0) ? $initialTotal : $FINAL_TOTAL_VARIANTS;

                // Use lock to prevent race conditions
                $lock = Cache::lock("progress_{$progressId}", 10);

                try {
                    if ($lock->get()) {
                        Cache::put($progressId, [
                            'total' => $finalTotal, // Always use initial total from controller
                            'processed' => $processedCount,
                            'errors' => $count_variants_error,
                            'status' => 'completed',
                            'current_variant' => null,
                        ], 3600);

                        $lock->release();
                    } else {
                        Cache::put($progressId, [
                            'total' => $finalTotal,
                            'processed' => $processedCount,
                            'errors' => $count_variants_error,
                            'status' => 'completed',
                            'current_variant' => null,
                        ], 3600);
                    }
                } catch (\Throwable $lockError) {
                    Cache::put($progressId, [
                        'total' => $finalTotal,
                        'processed' => $processedCount,
                        'errors' => $count_variants_error,
                        'status' => 'completed',
                        'current_variant' => null,
                    ], 3600);
                }
            }

            $demoResult->data = $demoObjects;
            $demoResult->summary = [
                'total_variants' => count($demoObjects),
                'updated' => $count_variants,
                'errors' => $count_variants_error,
                'demo_mode' => $demo,
                'progress_id' => $demo ? null : $progressId, // Return progress ID for non-demo mode
            ];

            // Collection title already set from fetched data or pre-obtained data

            // Save configuration to database (only if not in demo mode)
            if (!$demo) {
                try {
                    $configuration = CollectionCostConfiguration::findOrCreateForCollection($collection_id, $collectionTitle ?? 'Unknown Collection');
                    $configuration->update([
                        'collection_title' => $collectionTitle,
                        'profit_ranges' => $profit_ranges,
                        'expenses' => $expenses,
                        'rounding_option' => $rounding_option,
                    ]);
                } catch (\Throwable $th) {
                    Log::error("Failed to save collection configuration", [
                        'collection_id' => $collection_id,
                        'error' => $th->getMessage()
                    ]);
                    // Don't fail the whole operation if saving config fails
                }
            }

            // Save log entry (always, even in demo mode, to track all operations)
            try {
                $status = $count_variants_error > 0 && $count_variants === 0 ? 'failed' : 'success';
                $errorMessage = null;

                if ($status === 'failed' && $count_variants_error > 0) {
                    $errorMessage = "Failed to update {$count_variants_error} variant(s). No variants were updated successfully.";
                }

                CollectionCostPriceLog::create([
                    'collection_id' => $collection_id,
                    'collection_title' => $collectionTitle,
                    'status' => $status,
                    'executed_at' => now(),
                    'total_variants' => count($demoObjects),
                    'updated_variants' => $count_variants,
                    'errors' => $count_variants_error,
                    'error_message' => $errorMessage,
                    'demo_mode' => $demo,
                ]);
            } catch (\Throwable $th) {
                Log::error("Failed to save cost price log", [
                    'collection_id' => $collection_id,
                    'error' => $th->getMessage()
                ]);
                // Don't fail the whole operation if saving log fails
            }

            return $demoResult;
        } catch (\Throwable $th) {

            Log::error("Failed to update Shopify products variants", [
                'message' => $th->getMessage(),
                'trace' => $th->getTraceAsString(),
                'file' => $th->getFile(),
                'line' => $th->getLine(),
                'collection_id' => $collection_id ?? null,
                'progress_id' => $progressId ?? null,
            ]);

            // Try to save error log if we have collection_id
            try {
                $collectionId = $request->input('collection_id');
                if ($collectionId) {
                    CollectionCostPriceLog::create([
                        'collection_id' => $collectionId,
                        'collection_title' => 'Unknown Collection',
                        'status' => 'failed',
                        'executed_at' => now(),
                        'total_variants' => 0,
                        'updated_variants' => 0,
                        'errors' => 1,
                        'error_message' => $th->getMessage(),
                        'demo_mode' => $request->input('demo', false),
                    ]);
                }
            } catch (\Throwable $logError) {
                Log::error("Failed to save error log", [
                    'error' => $logError->getMessage()
                ]);
            }

            $response = [
                'status' => 500,
                'message' => $th->getMessage(),
                'data' => null,
            ];

            return $response;
        }
    }

    /**
     * Get profit for a given cost based on profit ranges
     *
     * @param float $cost
     * @param array $profitRanges
     * @return float
     */
    private function getProfitForCost(float $cost, array $profitRanges): float
    {
        foreach ($profitRanges as $range) {
            $minCost = (float)($range['min_cost'] ?? 0);
            $maxCost = (float)($range['max_cost'] ?? 0);
            $profitValue = (float)($range['profit'] ?? 0);
            $profitType = $range['profit_type'] ?? 'absolute';

            if ($cost >= $minCost && $cost <= $maxCost) {
                if ($profitType === 'percentage') {
                    return $cost * ($profitValue / 100);
                }
                return $profitValue;
            }
        }

        // If no range matches, return 0
        return 0;
    }


    /**
     * Get all product IDs from a specific collection
     *
     * @param Graphql $client
     * @param string $collectionId
     * @return array
     */
    /**
     * Execute bulk updates for a single product (price and compare price)
     */
    private function executeBulkUpdatesForProduct(string $productId, array $updates): void
    {
        try {
            $shop = env('SHOPIFY_DOMAIN');
            $token = env('SHOPIFY_ACCESS_TOKEN');

            if (!$shop || !$token) {
                return;
            }

            $client = new Graphql($shop, $token);

            // Prepare variants array with both price and compareAtPrice updates
            $variantsToUpdate = [];
            $variantMap = [];

            // First, collect all variant IDs and their updates
            foreach ($updates['price_updates'] as $update) {
                $variantId = $update['id'];
                if (!isset($variantMap[$variantId])) {
                    $variantMap[$variantId] = ['id' => $variantId];
                }
                $variantMap[$variantId]['price'] = $update['price'];
            }

            foreach ($updates['compare_price_updates'] as $update) {
                $variantId = $update['id'];
                if (!isset($variantMap[$variantId])) {
                    $variantMap[$variantId] = ['id' => $variantId];
                }
                $variantMap[$variantId]['compareAtPrice'] = $update['price'];
            }

            // Convert map to array
            $variantsToUpdate = array_values($variantMap);

            if (empty($variantsToUpdate)) {
                return;
            }

            // Single bulk update for all variants of this product
            $query = <<<QUERY
            mutation productVariantsBulkUpdate(\$productId: ID!, \$variants: [ProductVariantsBulkInput!]!) {
                productVariantsBulkUpdate(productId: \$productId, variants: \$variants) {
                    product {
                        id
                    }
                    productVariants {
                        id
                    }
                    userErrors {
                        field
                        message
                    }
                }
            }
            QUERY;

            $variables = [
                "productId" => $productId,
                "variants" => $variantsToUpdate,
            ];

            $response = $client->query(["query" => $query, "variables" => $variables]);
            $responseData = json_decode($response->getBody(), true);

            if (
                isset($responseData['data']['productVariantsBulkUpdate']['userErrors'])
                && count($responseData['data']['productVariantsBulkUpdate']['userErrors']) > 0
            ) {
                foreach ($responseData['data']['productVariantsBulkUpdate']['userErrors'] as $error) {
                    Log::error("Bulk update error for product {$productId}: " . $error['message']);
                }
            }
        } catch (\Throwable $th) {
            Log::error("Failed to execute bulk updates for product", [
                'product_id' => $productId,
                'message' => $th->getMessage(),
            ]);
        }
    }

    /**
     * Fetch collection data from Shopify
     * Public method so Controller can use it to get data before dispatching Job
     * 
     * @param string $collectionId
     * @return array ['products' => [], 'title' => string, 'excluded_product_ids' => []]
     */
    public function fetchCollectionData(string $collectionId): array
    {
        $shop = env('SHOPIFY_DOMAIN');
        $token = env('SHOPIFY_ACCESS_TOKEN');
        $excludedCollectionId = env('EXCLUDED_COLLECTION_ID');

        if (!$shop || !$token) {
            Log::error("Shopify domain or access token is not set in environment variables.");
            throw new \Exception('Shopify configuration missing');
        }

        $client = new Graphql($shop, $token);

        // Get list of products to exclude (if excluded collection is defined)
        $excludedProductIds = [];
        if ($excludedCollectionId) {
            $excludedProductIds = $this->getProductsFromCollection($client, $excludedCollectionId);
        }

        $query = <<<'GRAPHQL'
            query getProductsFromCollection($collection_id: ID!) {
                collection(id: $collection_id) {
                    id
                    title
                    products(first: 200) {
                        edges {
                            node {
                                id
                                title
                                status
                                variants(first: 200) {
                                    edges {
                                        node {
                                            id
                                            title
                                            price
                                            compareAtPrice
                                            inventoryItem {
                                                id
                                                unitCost {
                                                    amount
                                                }
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        GRAPHQL;

        $variables = [
            'collection_id' => $collectionId,
        ];

        $response = $client->query(["query" => $query, "variables" => $variables]);
        $data = json_decode($response->getBody(), true);

        return [
            'products' => $data['data']['collection']['products']['edges'] ?? [],
            'title' => $data['data']['collection']['title'] ?? 'Unknown Collection',
            'excluded_product_ids' => $excludedProductIds,
        ];
    }

    /**
     * Calculate total number of variants to process
     * Public method so Controller can use it to calculate total before initializing cache
     * 
     * @param array $productsShop
     * @param array $excludedProductIds
     * @return int
     */
    public function calculateTotalVariants(array $productsShop, array $excludedProductIds): int
    {
        $total = 0;

        foreach ($productsShop as $items) {
            $productId = $items['node']['id'];

            // Check if this product is in the excluded collection
            if (in_array($productId, $excludedProductIds)) {
                continue;
            }

            $variants = $items['node']['variants']['edges'] ?? [];

            foreach ($variants as $variant) {
                // Get cost price from Shopify data
                $inventoryItem = $variant['node']['inventoryItem'] ?? null;
                $unitCost = $inventoryItem['unitCost']['amount'] ?? null;
                $costPrice = $unitCost ? (float)$unitCost : 0;

                // Count only variants with valid cost price
                if ($costPrice > 0) {
                    $total++;
                }
            }
        }

        return $total;
    }

    private function getProductsFromCollection(Graphql $client, string $collectionId): array
    {
        try {
            $query = <<<'GRAPHQL'
            query getProductsFromCollection($collection_id: ID!) {
                collection(id: $collection_id) {
                    id
                    title
                    products(first: 250) {
                        edges {
                            node {
                                id
                            }
                        }
                    }
                }
            }
        GRAPHQL;

            $variables = [
                'collection_id' => $collectionId,
            ];

            $response = $client->query(["query" => $query, "variables" => $variables]);
            $data = json_decode($response->getBody(), true);

            $products = $data['data']['collection']['products']['edges'] ?? [];

            // Extract only the product IDs
            $productIds = array_map(function ($item) {
                return $item['node']['id'];
            }, $products);

            return $productIds;
        } catch (\Throwable $th) {
            Log::error("Error fetching excluded collection products", [
                'collection_id' => $collectionId,
                'error' => $th->getMessage()
            ]);
            return [];
        }
    }
}
