<?php

namespace App\Actions;

use App\Models\GoldPriceWeightRule;
use Illuminate\Support\Facades\Log;
use Shopify\Clients\Graphql;
use stdClass;

class GetMaterialDataByCollectionAction
{
    public function __invoke($collection_id)
    {
        try {
            $shop = env('SHOPIFY_DOMAIN');
            $token = env('SHOPIFY_ACCESS_TOKEN');

            if (!$shop || !$token) {
                Log::error("Shopify domain or access token is not set in environment variables.");
                return response()->json(['error' => 'Shopify configuration missing'], 500);
            }

            $client = new Graphql($shop, $token);

            // Query to get all products from collection first
            $query = <<<'GRAPHQL'
            query getProductsFromCollection($collection_id: ID!) {
                collection(id: $collection_id) {
                    id
                    title
                    products(first: 200) {
                        edges {
                            node {
                                id
                                title
                                metafield(namespace: "custom", key: "metal_purity") {
                                    key
                                    value
                                }
                                variants(first: 1) {
                                    edges {
                                        node {
                                            id
                                            price
                                            inventoryItem {
                                                measurement {
                                                    weight {
                                                        unit
                                                        value
                                                    }
                                                }
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        GRAPHQL;

            $variables = [
                'collection_id' => $collection_id,
            ];

            // Fetch all products from collection
            $response = $client->query(["query" => $query, "variables" => $variables]);
            $data = json_decode($response->getBody(), true);
            $allProducts = $data['data']['collection']['products']['edges'] ?? [];

            // Get all weight rules for this collection
            $weightRules = GoldPriceWeightRule::forCollection($collection_id)->get();

            // Find first 10k product whose weight is NOT in any rule
            $product10k = $this->findProductWithoutWeightRule($allProducts, '10k', $weightRules);

            // Find first 14k product whose weight is NOT in any rule
            $product14k = $this->findProductWithoutWeightRule($allProducts, '14k', $weightRules);

            // Process 10k product data
            $data10k = $this->processProductData($product10k);

            // Process 14k product data
            $data14k = $this->processProductData($product14k);

            $result = new stdClass();

            $result->count_10K = $data10k['found'];
            $result->price_10K = $data10k['price'];
            $result->weight_10K = $data10k['weight'];
            $result->current_price_10K = $data10k['price_per_gram'];

            $result->count_14K = $data14k['found'];
            $result->price_14K = $data14k['price'];
            $result->weight_14K = $data14k['weight'];
            $result->current_price_14K = $data14k['price_per_gram'];

            $result->total = count($allProducts); // Total products in collection

            return $result;
        } catch (\Throwable $th) {
            Log::error("Failed to fetch material data from collection", [
                'message' => $th->getMessage(),
                'trace' => $th->getTraceAsString(),
            ]);

            throw new \RuntimeException("An error occurred while fetching material data from collection.");
        }
    }

    /**
     * Check if a weight is within any of the provided weight rules
     *
     * @param float $weight The weight to check
     * @param \Illuminate\Database\Eloquent\Collection $weightRules Collection of weight rules
     * @return bool True if weight is within any rule, false otherwise
     */
    private function isWeightInRule(float $weight, $weightRules): bool
    {
        if ($weight <= 0) {
            return false; // No weight means no rule applies
        }

        foreach ($weightRules as $rule) {
            if ($rule->matchesWeight($weight)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Get the weight from the first variant of a product
     *
     * @param array $product Product node from Shopify GraphQL response
     * @return float Weight in grams, or 0 if not available
     */
    private function getProductWeight(array $product): float
    {
        if (empty($product['variants']['edges'])) {
            return 0;
        }

        $variant = $product['variants']['edges'][0]['node'];
        return floatval($variant['inventoryItem']['measurement']['weight']['value'] ?? 0);
    }

    /**
     * Check if a product matches the specified material type
     *
     * @param array $product Product node from Shopify GraphQL response
     * @param string $materialType '10k' or '14k'
     * @return bool True if product matches the material type
     */
    private function matchesMaterialType(array $product, string $materialType): bool
    {
        $metalPurity = strtolower($product['metafield']['value'] ?? '');

        if ($materialType === '10k') {
            return strpos($metalPurity, '10k') !== false;
        } elseif ($materialType === '14k') {
            return $metalPurity === '14k';
        }

        return false;
    }

    /**
     * Find the first product of a specific material type whose weight is NOT in any weight rule
     *
     * @param array $allProducts All products from the collection
     * @param string $materialType '10k' or '14k'
     * @param \Illuminate\Database\Eloquent\Collection $weightRules Collection of weight rules
     * @return array|null Product node if found, null otherwise
     */
    private function findProductWithoutWeightRule(array $allProducts, string $materialType, $weightRules): ?array
    {
        foreach ($allProducts as $productEdge) {
            $product = $productEdge['node'] ?? null;

            if (!$product || !$this->matchesMaterialType($product, $materialType)) {
                continue;
            }

            $weight = $this->getProductWeight($product);

            // Only use this product if its weight is NOT in any rule
            if (!$this->isWeightInRule($weight, $weightRules)) {
                return $product;
            }
        }

        return null;
    }

    /**
     * Process product data and calculate price per gram
     *
     * @param array|null $product Product node from Shopify, or null if not found
     * @return array Contains: found (bool), price (float), weight (float), price_per_gram (float)
     */
    private function processProductData(?array $product): array
    {
        if (!$product) {
            return [
                'found' => false,
                'price' => 0,
                'weight' => 0,
                'price_per_gram' => 0,
            ];
        }

        try {
            $price = 0;
            $weight = 0;

            if (!empty($product['variants']['edges'])) {
                $variant = $product['variants']['edges'][0]['node'];
                $price = floatval($variant['price'] ?? 0);
                $weight = floatval($variant['inventoryItem']['measurement']['weight']['value'] ?? 0);
            }

            $pricePerGram = ($weight > 0) ? ($price / $weight) : $price;

            return [
                'found' => true,
                'price' => $price,
                'weight' => $weight,
                'price_per_gram' => $pricePerGram,
            ];
        } catch (\Throwable $th) {
            $materialType = $this->matchesMaterialType($product, '10k') ? '10k' : '14k';
            Log::error("Error processing {$materialType} product: " . $th->getMessage());

            return [
                'found' => false,
                'price' => 0,
                'weight' => 0,
                'price_per_gram' => 0,
            ];
        }
    }
}
