<?php

namespace App\Actions;

use App\Models\GoldPrice;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class FetchGoldPriceAction
{
    public function __invoke()
    {
        try {
            $apiKey = config('services.metalprice.api_key');
            $baseUrl = 'https://api.metalpriceapi.com/v1/latest';

            if (!$apiKey) {
                Log::error('METALPRICE_API_KEY not configured');
                throw new \RuntimeException('METALPRICE_API_KEY not found in environment variables.');
            }

            $response = Http::get($baseUrl, [
                'api_key' => $apiKey,
                'base' => 'USD',
                'currencies' => 'EUR,XAU,XAG'
            ]);

            if (!$response->successful()) {
                $body = $response->body();
                Log::error('Metal Price API error', [
                    'status' => $response->status(),
                    'body' => $body
                ]);
                throw new \RuntimeException('Failed to fetch gold price from API. Status: ' . $response->status() . ' - Response: ' . $body);
            }

            $data = $response->json();

            if (!isset($data['success']) || !$data['success']) {
                $errorMessage = $data['error']['type'] ?? 'Unknown error';
                $errorInfo = $data['error']['info'] ?? json_encode($data);
                Log::error('Metal Price API unsuccessful response', ['data' => $data]);
                throw new \RuntimeException('API returned unsuccessful response. Error: ' . $errorMessage . ' - Info: ' . $errorInfo);
            }

            // Extraer el precio del oro (XAU)
            $xauPrice = $data['rates']['XAU'] ?? null;
            $apiTimestamp = $data['timestamp'] ?? now()->timestamp;

            if (!$xauPrice) {
                Log::error('XAU price not found in API response', ['data' => $data]);
                throw new \RuntimeException('Gold price (XAU) not found in API response.');
            }

            // Calcular el precio en USD por onza (XAU es la inversa)
            $pricePerOunce = 1 / $xauPrice;

            // Guardar en la base de datos
            $goldPrice = GoldPrice::create([
                'price' => $pricePerOunce,
                'api_timestamp' => now()->setTimestamp($apiTimestamp),
            ]);

            Log::info('Gold price fetched and saved', [
                'price' => $pricePerOunce,
                'timestamp' => $apiTimestamp
            ]);

            return $goldPrice;
        } catch (\Throwable $th) {
            Log::error("Failed to fetch gold price", [
                'message' => $th->getMessage(),
                'trace' => $th->getTraceAsString(),
            ]);

            throw new \RuntimeException("An error occurred while fetching gold price: " . $th->getMessage());
        }
    }
}
